<?php
/**
 * Minimalist Storage Tool
 * Features: Sequential Text & Image Storage, Hash Deduplication, Full-Width Gallery
 */

// --- CONFIGURATION ---
$dirs = [
    'text'     => 'text',      // Sequential URL lists
    'img'      => 'img',       // Sequential JPGs
    'img_text' => 'img_text',  // Image descriptions
    'tmp_img'  => 'tmp_img'    // Hash files for deduplication
];

$maxImgSize    = 500 * 1024;
$itemsPerPage  = 10;
$status        = "";

// --- SETUP ---
foreach ($dirs as $dir) {
    if (!is_dir($dir)) mkdir($dir, 0777, true);
}

// --- LOGIC: POST HANDLERS ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // 1. URL LIST STORAGE
    if (!empty($_POST['urls'])) {
        $urlContent = trim($_POST['urls']);
        $count = 1;
        while (file_exists("{$dirs['text']}/$count.txt")) { $count++; }
        
        $fp = fopen("{$dirs['text']}/$count.txt", 'w');
        if ($fp && flock($fp, LOCK_EX)) {
            fwrite($fp, $urlContent);
            flock($fp, LOCK_UN);
            fclose($fp);
            $status = "<a href='text/{$count}.txt'>URL list saved as {$count}.txt</a>";
        }
    }

    // 2. IMAGE & DESCRIPTION STORAGE
    if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $desc = substr($_POST['desc'] ?? '', 0, 500);
        $tmp  = $_FILES['image']['tmp_name'];
        $ext  = strtolower(pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION));
        
        if ($ext !== 'jpg') {
            $status = "Error: Only .jpg allowed.";
        } elseif ($_FILES['image']['size'] > $maxImgSize) {
            $status = "Error: File too large (>500KB).";
        } else {
            $hash = hash_file('sha256', $tmp);
            if (file_exists("{$dirs['tmp_img']}/$hash")) {
                $status = "Error: Duplicate image detected.";
            } else {
                $i = 1;
                while (file_exists("{$dirs['img']}/$i.jpg")) { $i++; }

                if (move_uploaded_file($tmp, "{$dirs['img']}/$i.jpg")) {
                    touch("{$dirs['tmp_img']}/$hash");
                    file_put_contents("{$dirs['img_text']}/$i.txt", $desc);
                    $status = "Stored successfully as item #$i";
                }
            }
        }
    }
}

// --- LOGIC: PAGINATION ---
$images = glob("{$dirs['img']}/*.jpg");
natsort($images);
$images = array_reverse(array_values($images));

$total       = count($images);
$totalPages  = ceil($total / $itemsPerPage);
$page        = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$pagedItems  = array_slice($images, ($page - 1) * $itemsPerPage, $itemsPerPage);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Media Storage</title>
    <style>
        :root { --border: #000; --soft-grey: #f4f4f4; }
        body { font-family: "Inter", "Helvetica", sans-serif; margin: 0; padding: 40px 20px; background: #fff; color: #000; line-height: 1.6; }
        .container { max-width: 650px; margin: 0 auto; }
        
        header { border-bottom: 3px solid var(--border); margin-bottom: 50px; padding-bottom: 10px; }
        h1 { font-size: 1.5rem; letter-spacing: -0.02em; margin: 0; }
        h2 { font-size: 0.75rem; text-transform: uppercase; letter-spacing: 0.1em; color: #666; margin-bottom: 15px; }
        
        section { margin-bottom: 60px; }
        .status { background: #000; color: #fff; padding: 10px; margin-bottom: 30px; font-size: 0.85rem; text-align: center; }
        
        textarea, input[type="file"] { width: 100%; border: 1px solid #ccc; padding: 12px; margin-bottom: 15px; box-sizing: border-box; font-family: inherit; }
        button { background: #000; color: #fff; border: none; padding: 15px; width: 100%; font-weight: bold; cursor: pointer; text-transform: uppercase; font-size: 0.75rem; transition: opacity 0.2s; }
        button:hover { opacity: 0.8; }

        /* Gallery Styles */
        .archive-item { margin-bottom: 80px; }
        .archive-item a { display: block; border: 1px solid #eee; overflow: hidden; }
        .archive-item img { width: 100%; display: block; height: auto; transition: transform 0.3s ease; }
        .archive-item img:hover { transform: scale(1.01); }
        
        .description-box { padding-top: 15px; }
        .item-meta { font-weight: 800; font-size: 0.8rem; margin-bottom: 5px; display: block; }
        .item-text { font-size: 0.95rem; color: #333; }

        /* Pagination */
        .pagination { display: flex; gap: 8px; justify-content: center; margin-top: 50px; border-top: 1px solid #eee; padding-top: 30px; }
        .pagination a { text-decoration: none; color: #000; padding: 8px 14px; border: 1px solid #000; font-size: 0.8rem; font-weight: bold; }
        .pagination a.active { background: #000; color: #fff; }
    </style>
</head>
<body>

<div class="container">
    <header>
        <h1>VAULT_01</h1>
    </header>

    <?php if ($status): ?>
        <div class="status"><?php echo $status; ?></div>
    <?php endif; ?>

    <section>
        <h2>01_Store_URLs</h2>
        <form method="POST">
            <textarea name="urls" rows="4" placeholder="Paste link list..."></textarea>
            <button type="submit">Archive Text</button>
        </form>
    </section>

    <section>
        <h2>02_Upload_JPG</h2>
        <form method="POST" enctype="multipart/form-data">
            <input type="file" name="image" accept=".jpg" required>
            <textarea name="desc" placeholder="Add description (max 500 characters)..." maxlength="500"></textarea>
            <button type="submit">Archive Image</button>
        </form>
    </section>

    <section>
        <h2>03_Gallery</h2>
        <?php if (empty($pagedItems)): ?>
            <p style="color:#999">No items found.</p>
        <?php else: ?>
            <?php foreach ($pagedItems as $img): 
                $id = pathinfo($img, PATHINFO_FILENAME);
                $txtPath = "{$dirs['img_text']}/$id.txt";
                $textData = file_exists($txtPath) ? file_get_contents($txtPath) : "";
            ?>
                <div class="archive-item">
                    <a href="<?php echo $img; ?>" target="_blank">
                        <img src="<?php echo $img; ?>" alt="Item <?php echo $id; ?>">
                    </a>
                    <div class="description-box">
                        <span class="item-meta">ENTRY_#<?php echo str_pad($id, 3, '0', STR_PAD_LEFT); ?></span>
                        <div class="item-text"><?php echo nl2br(htmlspecialchars($textData)); ?></div>
                    </div>
                </div>
            <?php endforeach; ?>

            <div class="pagination">
                <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                    <a href="?page=<?php echo $i; ?>" class="<?php echo ($i == $page) ? 'active' : ''; ?>">
                        <?php echo $i; ?>
                    </a>
                <?php endfor; ?>
            </div>
        <?php endif; ?>
    </section>
</div>

</body>
</html>