<?php
// Get user search input
$search = $_GET['search'] ?? '';

function getBaseUrl($url) {
    // Extract base URL without the current file
    $parts = parse_url($url);
    $scheme = $parts['scheme'] ?? 'http';
    $host = $parts['host'] ?? '';
    $port = isset($parts['port']) ? ':' . $parts['port'] : '';
    return "{$scheme}://{$host}{$port}";
}

// Read list of server URLs from servers.txt
// Ensure servers.txt exists in the same directory
$servers = [];
if (file_exists('servers.txt')) {
    $servers = file('servers.txt', FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
}

$results = [];

if ($search && !empty($servers)) {
    foreach ($servers as $serverUrl) {
        $baseUrl = getBaseUrl($serverUrl);
        // Build JSON file path based on user input at /json_search/ folder
        $jsonUrl = rtrim($baseUrl, '/') . '/json_search/' . urlencode($search) . '.json';

        // Check if file exists by trying to fetch
        $content = @file_get_contents($jsonUrl);
        if ($content === false) {
            continue;
        }

        // Decode JSON content expected to be an array of objects
        $dataArray = json_decode("[$content]", true);
        if (!is_array($dataArray)) {
            continue;
        }

        // Collect entries for this server
        foreach ($dataArray as $entry) {
            $results[] = [
                'title' => $entry['title'] ?? 'Unknown',
                'hash' => $entry['filename'] ?? 'Unknown',
                'ping' => '', // No ping available in JSON
                'url' => $jsonUrl,
                'baseUrl' => $baseUrl
            ];
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>JSON Search Results</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #f5f5f5;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: flex-start; /* Changed to flex-start to handle long result lists */
            padding: 20px;
            padding-top: 100px; /* Increased slightly to clear nav */
        }

        .nav-bar {
            position: fixed;
            top: 0; left: 0;
            width: 100%; height: 60px;
            background: #fff;
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0 40px;
            box-shadow: 0 1px 5px rgba(0,0,0,0.05);
            z-index: 1000;
        }

        .nav-logo { font-weight: 600; font-size: 18px; color: #333; text-decoration: none; }
        .nav-links { display: flex; gap: 30px; }
        .nav-links a { text-decoration: none; color: #666; font-size: 14px; text-transform: uppercase; transition: 0.3s; }
        .nav-links a:hover { color: #000; }
        
        .container { width: 100%; max-width: 900px; }
        .form-wrapper { background: white; padding: 50px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.05); margin-bottom: 30px; }
        h1 { text-align: center; margin-bottom: 40px; color: #333; font-weight: 300; font-size: 32px; }
        
        form { display: flex; flex-direction: column; gap: 20px; }
        input[type="text"] {
            padding: 20px 25px; font-size: 18px; border: 2px solid #e0e0e0; border-radius: 4px;
            outline: none; transition: 0.3s; height: 70px; width: 100%;
        }
        input[type="text"]:focus { border-color: #333; }
        
        .radio-group { display: flex; gap: 20px; justify-content: center; padding: 10px 0; }
        .radio-option { display: flex; align-items: center; gap: 8px; font-size: 18px; cursor: pointer; }
        
        button {
            padding: 20px 40px; font-size: 18px; background: #333; color: white; border: none;
            border-radius: 4px; cursor: pointer; transition: 0.3s; height: 70px;
        }
        button:hover { background: #555; }
        
        .message { padding: 20px; border-radius: 4px; margin-bottom: 20px; text-align: center; }
        .message.success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .message.error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .message.info { background: #d1ecf1; color: #0c5460; border: 1px solid #bee5eb; }

        /* Additional Styles for Results Table matching the theme */
        .results-wrapper {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            width: 100%;
            overflow-x: auto;
        }
        table { width: 100%; border-collapse: collapse; }
        th, td { padding: 15px; text-align: left; border-bottom: 1px solid #eee; }
        th { font-weight: 600; color: #333; text-transform: uppercase; font-size: 14px; letter-spacing: 0.5px; }
        td { color: #555; font-size: 15px; }
        tr:last-child td { border-bottom: none; }
        a.result-link { color: #333; text-decoration: none; font-weight: 500; border-bottom: 1px dotted #999; }
        a.result-link:hover { color: #000; border-bottom: 1px solid #000; }

        @media (max-width: 768px) {
            .nav-bar { padding: 0 20px; }
            .nav-links { gap: 15px; }
            .form-wrapper { padding: 30px 20px; }
            th, td { padding: 10px; font-size: 14px; }
        }
    </style>
</head>
<body>

    <nav class="nav-bar">
        <a href="?" class="nav-logo">JSON Search</a>
        <div class="nav-links">
            <a href="?">Home</a>
            <a href="#">About</a>
        </div>
    </nav>

    <div class="container">
        
        <div class="form-wrapper">
            <h1>Search JSON Files</h1>
            <form method="get" action="">
                <input type="text" id="search" name="search" placeholder="Enter file name..." value="<?= htmlspecialchars($search) ?>" required />
                <button type="submit">Search</button>
            </form>
        </div>

        <?php if ($search): ?>
            
            <?php if (empty($servers)): ?>
                <div class="message error">
                    Error: <strong>servers.txt</strong> not found or empty.
                </div>
            <?php elseif (empty($results)): ?>
                <div class="message info">
                    No entries found for "<strong><?= htmlspecialchars($search) ?></strong>".
                </div>
            <?php else: ?>
                <div class="results-wrapper">
                    <table>
                        <thead>
                            <tr>
                                <th>Title</th>
                                <th>Hash</th>
                                <th>Ping</th>
                                <th>Source URL</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($results as $row): ?>
                                <tr>
                                    <td>
                                        <a class="result-link" href="<?= htmlspecialchars($row['baseUrl'] . '/redirect.html?hash=' . $row['hash']) ?>" target="_blank" rel="noopener noreferrer">
                                            <?= htmlspecialchars($row['title']) ?>
                                        </a>
                                    </td>
                                    <td>
                                        <?= htmlspecialchars($row['hash']) ?>
                                    </td>
                                    <td><?= htmlspecialchars($row['ping']) ?></td>
                                    <td>
                                        <a class="result-link" href="<?= htmlspecialchars($row['baseUrl']) ?>" target="_blank" rel="noopener noreferrer">
                                            <?= htmlspecialchars($row['url']) ?>
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>

        <?php endif; ?>

    </div>

</body>
</html>