﻿<?php
// Configuration
$baseDir = 'rhyzom';
$tmpDir = 'tmp_rhyzom';
$message = "";
$currentView = $_GET['view'] ?? 'landing';

// Helper: Securely get folder list for the browser
function getCategories($path) {
    if (!is_dir($path)) return [];
    return array_filter(scandir($path), function($item) use ($path) {
        return is_dir($path . '/' . $item) && !in_array($item, ['.', '..']);
    });
}

// --- Processing Logic: File Writing ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['password'])) {
    $currentView = 'app';
    
    // 1. Process Categories (Single field exploded by comma)
    $categories = [];
    if (!empty($_POST['categories_raw'])) {
        // Split by comma or space
        $rawArray = preg_split('/[,\s]+/', $_POST['categories_raw']);
        foreach (array_slice($rawArray, 0, 5) as $cat) {
            $cat = trim($cat);
            // Validation: alphanumeric + underscore, max 12 chars
            if (preg_match('/^[a-zA-Z0-9_]{1,12}$/', $cat) && !empty($cat)) {
                $categories[] = $cat;
            }
        }
    }

    // 2. Process Password -> SHA-256 Hash for filename
    $password = substr($_POST['password'], 0, 50);
    $fileHash = hash('sha256', $password);

    // 3. Content (Now allows any input)
    $content = $_POST['content'] ?? '';

    // 4. File Writing Logic
    if (empty($categories)) {
        $message = "Error: Please enter at least one valid category name.";
    } elseif (empty($password)) {
        $message = "Error: Password is required to generate the file hash.";
    } else {
        foreach ($categories as $folderName) {
            $targetFolder = $baseDir . DIRECTORY_SEPARATOR . $folderName;
            if (!is_dir($targetFolder)) {
                mkdir($targetFolder, 0755, true);
            }
            $filePath = $targetFolder . DIRECTORY_SEPARATOR . $fileHash . ".txt";
            
            // Write using fopen ("w" overwrites)
            $handle = fopen($filePath, "w");
            if ($handle) {
                fwrite($handle, $content);
                fclose($handle);
            }
        }
        $message = "Success: Content saved in " . count($categories) . " category folders.";
    }
}

// --- Tracking & Reading Logic ---
$fileContent = "";
if ($currentView === 'read' && isset($_GET['cat']) && isset($_GET['file'])) {
    $safeCat = basename($_GET['cat']);
    $safeFile = basename($_GET['file']);
    $targetPath = $baseDir . "/$safeCat/$safeFile";

    if (file_exists($targetPath)) {
        $fileContent = file_get_contents($targetPath);
        
        // Access Tracking Logic: Hash of user IP
        $userIpHash = hash('sha256', $_SERVER['REMOTE_ADDR']);
        $trackFolder = $tmpDir . DIRECTORY_SEPARATOR . $safeFile;
        
        if (!is_dir($trackFolder)) mkdir($trackFolder, 0755, true);
        
        // Write empty file as an access token
        $trackHandle = fopen($trackFolder . DIRECTORY_SEPARATOR . $userIpHash, "w");
        if ($trackHandle) fclose($trackHandle);
    } else {
        $message = "Error: File not found.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rhyzom | Secure Data Node</title>
    <style>
        :root { --primary: #6366f1; --secondary: #4338ca; --bg: #fdfdfd; --text: #1e293b; }
        body { font-family: 'Segoe UI', system-ui, sans-serif; background: var(--bg); color: var(--text); margin: 0; line-height: 1.5; }
        .nav { background: #fff; padding: 1rem 2rem; display: flex; justify-content: space-between; align-items: center; border-bottom: 1px solid #e2e8f0; }
        .nav .logo { font-weight: 900; font-size: 1.4rem; color: var(--primary); text-decoration: none; letter-spacing: -1px; }
        .container { max-width: 800px; margin: 2rem auto; padding: 0 1.5rem; }
        .card { background: #fff; padding: 2rem; border-radius: 16px; border: 1px solid #e2e8f0; box-shadow: 0 4px 6px -1px rgba(0,0,0,0.05); }
        h1, h2 { color: #0f172a; }
        .field-group { margin-bottom: 1.25rem; }
        label { display: block; font-weight: 600; margin-bottom: 0.5rem; font-size: 0.9rem; }
        input[type="text"], input[type="password"], textarea { 
            width: 100%; padding: 0.8rem; border: 1px solid #cbd5e1; border-radius: 8px; box-sizing: border-box; font-size: 1rem; transition: border 0.2s;
        }
        input:focus, textarea:focus { outline: none; border-color: var(--primary); }
        textarea { height: 180px; font-family: monospace; resize: vertical; }
        .btn { display: inline-block; padding: 0.8rem 1.5rem; border-radius: 8px; font-weight: 600; text-decoration: none; cursor: pointer; border: none; transition: all 0.2s; text-align: center; }
        .btn-primary { background: var(--primary); color: #fff; width: 100%; }
        .btn-primary:hover { background: var(--secondary); }
        .btn-ghost { color: var(--text); font-size: 0.9rem; }
        .msg { padding: 1rem; border-radius: 8px; margin-bottom: 1.5rem; font-weight: 500; text-align: center; }
        .error { background: #fef2f2; color: #991b1b; }
        .success { background: #f0fdf4; color: #166534; }
        .grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(180px, 1fr)); gap: 1rem; }
        .item-link { display: block; padding: 1rem; background: #f8fafc; border-radius: 10px; text-decoration: none; color: var(--text); border: 1px solid #e2e8f0; text-align: center; }
        .item-link:hover { border-color: var(--primary); background: #f1f5f9; }
        pre { background: #0f172a; color: #e2e8f0; padding: 1.5rem; border-radius: 10px; overflow-x: auto; white-space: pre-wrap; font-size: 0.9rem; }
    </style>
</head>
<body>

<nav class="nav">
    <a href="?view=landing" class="logo">RHYZOM</a>
    <div>
        <a href="?view=browse" class="btn btn-ghost">Browse Nodes</a>
        <a href="?view=app" class="btn btn-primary" style="width: auto;">+ New Entry</a>
    </div>
</nav>

<div class="container">
    <?php if ($message): ?>
        <div class="msg <?php echo strpos($message, 'Error') === 0 ? 'error' : 'success'; ?>"><?php echo $message; ?></div>
    <?php endif; ?>

    <?php if ($currentView === 'landing'): ?>
        <section style="text-align: center; padding: 3rem 0;">
            <h1>Content-Addressed Knowledge</h1>
            <p style="color: #64748b; font-size: 1.1rem; max-width: 500px; margin: 0 auto 2rem;">
                A simple, local-first system for categorizing data using cryptographic keys and unique access tracking.
            </p>
            <a href="?view=app" class="btn btn-primary" style="width: auto; padding: 1rem 2.5rem;">Launch Dashboard</a>
        </section>

    <?php elseif ($currentView === 'browse'): ?>
        <div class="card">
            <h2>Active Categories</h2>
            <div class="grid">
                <?php 
                $cats = getCategories($baseDir);
                foreach ($cats as $c): ?>
                    <a href="?view=list&cat=<?php echo urlencode($c); ?>" class="item-link">📁 <?php echo htmlspecialchars($c); ?></a>
                <?php endforeach; ?>
                <?php if (empty($cats)) echo "<p>No categories detected yet.</p>"; ?>
            </div>
        </div>

    <?php elseif ($currentView === 'list'): ?>
        <div class="card">
            <h2>Files in: <?php echo htmlspecialchars($_GET['cat']); ?></h2>
            <div class="grid">
                <?php 
                $safeCat = basename($_GET['cat']);
                $files = array_diff(scandir($baseDir."/".$safeCat), ['.', '..']);
                foreach ($files as $f): ?>
                    <a href="?view=read&cat=<?php echo urlencode($safeCat); ?>&file=<?php echo urlencode($f); ?>" class="item-link">📄 <?php echo substr($f, 0, 12); ?>...</a>
                <?php endforeach; ?>
            </div>
            <p><a href="?view=browse" class="btn btn-ghost">&larr; Back to Categories</a></p>
        </div>

    <?php elseif ($currentView === 'read'): ?>
        <div class="card">
            <h2>Data View</h2>
            <p style="font-size: 0.8rem; color: #64748b;">SHA-256 ID: <?php echo htmlspecialchars($_GET['file']); ?></p>
            <pre><?php echo htmlspecialchars($fileContent); ?></pre>
            
            <?php
                $trackPath = $tmpDir . '/' . basename($_GET['file']);
                $views = is_dir($trackPath) ? count(array_diff(scandir($trackPath), ['.', '..'])) : 0;
            ?>
            <div style="margin-top: 1rem; padding: 1rem; background: #f1f5f9; border-radius: 8px; display: flex; justify-content: space-between; align-items: center;">
                <span><strong>Unique Reach:</strong> <?php echo $views; ?> IPs</span>
                <a href="?view=list&cat=<?php echo urlencode($_GET['cat']); ?>" class="btn btn-primary" style="width: auto;">Close</a>
            </div>
        </div>

    <?php elseif ($currentView === 'app'): ?>
        <div class="card">
            <h2>Create New Node</h2>
            <form method="POST">
                <div class="field-group">
                    <label>Categories (Separate by commas, e.g. "finance, btc, private")</label>
                    <input type="text" name="categories_raw" placeholder="Enter up to 5 categories..." required>
                </div>

                <div class="field-group">
                    <label>Content (Any text, metadata, or links)</label>
                    <textarea name="content" placeholder="Type or paste your data here..."></textarea>
                </div>

                <div class="field-group">
                    <label>Encryption Password (Creates the unique SHA-256 filename)</label>
                    <input type="password" name="password" placeholder="Your secret password" required>
                </div>

                <button type="submit" class="btn btn-primary">Hash and Save Content</button>
            </form>
        </div>
    <?php endif; ?>
</div>

</body>
</html>