<?php
/**
 * Rhyzom Project - Refactored Version
 * Features: Password-based hashing, Single-field category input, Any content textarea.
 */

$page = isset($_GET['page']) ? $_GET['page'] : 'landing';
$message = "";
$messageType = "";
$baseDir = "rhyzom";
$tmpDir = "tmp_rhyzom";

// --- LOGIC SECTION: DATA PROCESSING ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && $page == 'app') {
    try {
        // 1. Process Categories (Single field exploded by comma)
        $rawCategories = isset($_POST['categories']) ? $_POST['categories'] : '';
        $categoriesArray = array_filter(array_map('trim', explode(',', $rawCategories)));
        
        if (count($categoriesArray) > 5) throw new Exception("Error: Maximum of 5 categories allowed.");
        if (empty($categoriesArray)) throw new Exception("Error: At least one category is required.");

        foreach ($categoriesArray as $cat) {
            if (strlen($cat) > 12) throw new Exception("Error: Category '$cat' exceeds 12 characters.");
            if (!preg_match('/^[a-zA-Z0-9_]+$/', $cat)) {
                throw new Exception("Error: Category '$cat' contains invalid characters. Use letters, numbers, and '_' only.");
            }
        }

        // 2. Process Password (formerly signature)
        $password = isset($_POST['password']) ? trim($_POST['password']) : '';
        if (strlen($password) > 50) throw new Exception("Error: Password must be maximum 50 characters.");
        if (empty($password)) throw new Exception("Error: Password is required.");
        
        $passwordHash = hash('sha256', $password);

        // 3. Process Textarea (Any content allowed)
        $content = isset($_POST['content']) ? $_POST['content'] : '';
        if (empty($content)) throw new Exception("Error: Content cannot be empty.");

        // 4. Save Files using fopen
        if (!is_dir($baseDir)) mkdir($baseDir, 0777, true);

        foreach ($categoriesArray as $category) {
            $catPath = $baseDir . DIRECTORY_SEPARATOR . $category;
            if (!is_dir($catPath)) mkdir($catPath, 0777, true);
            
            $filePath = $catPath . DIRECTORY_SEPARATOR . $passwordHash . ".txt";
            $handle = fopen($filePath, "w");
            if ($handle) {
                fwrite($handle, $content);
                fclose($handle);
            } else {
                throw new Exception("Error opening file for writing at: $filePath");
            }
        }

        $message = "Success! Data processed and saved securely.";
        $messageType = "success";

    } catch (Exception $e) {
        $message = $e->getMessage();
        $messageType = "error";
    }
}

// --- LOGIC SECTION: FILE EXPLORER & ACCESS COUNTING ---
$selectedCategory = isset($_GET['category']) ? $_GET['category'] : '';
$viewFile = isset($_GET['view']) ? $_GET['view'] : '';

if ($viewFile && $selectedCategory) {
    $userIp = $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    $ipHash = hash('sha256', $userIp);
    $fileNameOnly = basename($viewFile, ".txt");
    $accessLogDir = $tmpDir . DIRECTORY_SEPARATOR . $fileNameOnly;
    
    if (!is_dir($accessLogDir)) mkdir($accessLogDir, 0777, true);
    $logFile = $accessLogDir . DIRECTORY_SEPARATOR . $ipHash;
    $handle = fopen($logFile, "w");
    if ($handle) fclose($handle);
}

$availableCategories = is_dir($baseDir) ? array_diff(scandir($baseDir), array('.', '..')) : [];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rhyzom - Secure Storage</title>
    <style>
        :root { --primary: #2563eb; --primary-hover: #1d4ed8; --success: #059669; --error: #dc2626; --bg: #f8fafc; --text: #1e293b; --card: #ffffff; }
        body { font-family: 'Inter', sans-serif; background: var(--bg); color: var(--text); margin: 0; line-height: 1.6; }
        .container { max-width: 1000px; margin: 0 auto; padding: 2rem; }
        nav { display: flex; justify-content: space-between; align-items: center; padding: 1rem 0; border-bottom: 1px solid #e2e8f0; margin-bottom: 2rem; }
        .logo { font-size: 1.5rem; font-weight: 800; color: var(--primary); text-decoration: none; }
        .nav-links a { text-decoration: none; color: var(--text); font-weight: 500; margin-left: 1.5rem; }
        .card { background: var(--card); padding: 2rem; border-radius: 1rem; box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1); margin-bottom: 2rem; }
        .btn { display: inline-block; padding: 0.75rem 1.5rem; border-radius: 0.5rem; text-decoration: none; font-weight: 600; cursor: pointer; border: none; transition: all 0.2s; }
        .btn-primary { background: var(--primary); color: white; }
        .btn-primary:hover { background: var(--primary-hover); }
        .btn-success { background: var(--success); color: white; width: 100%; }
        .form-group { margin-bottom: 1.5rem; }
        label { display: block; margin-bottom: 0.5rem; font-weight: 600; }
        input, textarea { width: 100%; padding: 0.8rem; border: 1px solid #cbd5e1; border-radius: 0.5rem; box-sizing: border-box; font-family: inherit; font-size: 1rem; }
        textarea { height: 200px; resize: vertical; }
        small { color: #64748b; display: block; margin-top: 0.4rem; font-size: 0.85rem; }
        .grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 1rem; }
        .cat-item { padding: 1rem; background: #f1f5f9; border-radius: 0.5rem; text-align: center; text-decoration: none; color: var(--text); font-weight: 600; }
        .cat-item:hover, .cat-item.active { background: var(--primary); color: white; }
        .file-list { list-style: none; padding: 0; }
        .file-item { display: flex; justify-content: space-between; align-items: center; padding: 1rem; border-bottom: 1px solid #f1f5f9; }
        .file-link { color: var(--primary); text-decoration: none; font-family: monospace; }
        .alert { padding: 1rem; border-radius: 0.5rem; margin-bottom: 1.5rem; font-weight: 500; }
        .alert-success { background: #dcfce7; color: #166534; border: 1px solid #bbf7d0; }
        .alert-error { background: #fee2e2; color: #991b1b; border: 1px solid #fecaca; }
        .access-count { font-size: 0.8rem; color: #64748b; background: #e2e8f0; padding: 0.25rem 0.75rem; border-radius: 1rem; }
    </style>
</head>
<body>

<div class="container">
    <nav>
        <a href="?page=landing" class="logo">RHYZOM</a>
        <div class="nav-links">
            <a href="?page=landing">Home</a>
            <a href="?page=app">Processor</a>
            <a href="?page=explorer">Explorer</a>
        </div>
    </nav>

    <?php if ($page == 'landing'): ?>
        <section style="text-align: center; padding: 4rem 0;">
            <h1 style="font-size: 3rem; margin-bottom: 1rem;">Secure Content Vault</h1>
            <p style="font-size: 1.25rem; color: #64748b; max-width: 700px; margin: 0 auto 2.5rem;">
                Store any content, categorized and protected by SHA-256 password hashing. 
                Simple, private, and efficient.
            </p>
            <div>
                <a href="?page=app" class="btn btn-primary" style="padding: 1rem 2.5rem; font-size: 1.1rem;">Launch Processor</a>
            </div>
        </section>

        <section class="grid" style="margin-top: 2rem;">
            <div class="card">
                <h3 style="color: var(--primary);">Password Protection</h3>
                <p>Your files are named using the hash of your password. Only those who know the password can identify the file.</p>
            </div>
            <div class="card">
                <h3 style="color: var(--primary);">Multi-Category</h3>
                <p>Input multiple categories separated by commas. Your content will be mirrored across all of them.</p>
            </div>
            <div class="card">
                <h3 style="color: var(--primary);">Access Tracking</h3>
                <p>Monitor unique views per file based on visitor IP hashes, stored in a temporary directory.</p>
            </div>
        </section>

    <?php elseif ($page == 'app'): ?>
        <div class="card">
            <h2 style="margin-top: 0;">Data Processor</h2>
            <?php if ($message): ?><div class="alert alert-<?php echo $messageType; ?>"><?php echo $message; ?></div><?php endif; ?>
            
            <form method="POST" action="?page=app">
                <div class="form-group">
                    <label for="categories">Categories</label>
                    <input type="text" id="categories" name="categories" placeholder="e.g. work, private, backup" required>
                    <small>Separate by commas. Max 5 categories. Max 12 chars each (A-Z, 0-9, _).</small>
                </div>

                <div class="form-group">
                    <label for="password">Password</label>
                    <input type="text" id="password" name="password" maxlength="50" placeholder="Enter your password" required>
                    <small>This generates the SHA-256 filename. Max 50 characters.</small>
                </div>

                <div class="form-group">
                    <label for="content">Content</label>
                    <textarea id="content" name="content" placeholder="Enter any text, links, or data here..." required></textarea>
                    <small>Any content is allowed. No format restrictions.</small>
                </div>

                <button type="submit" class="btn btn-success">Process & Save Content</button>
            </form>
        </div>

    <?php elseif ($page == 'explorer'): ?>
        <div class="card">
            <h2 style="margin-top: 0;">File Explorer</h2>
            <form method="GET" action="" style="margin-bottom: 2rem; display: flex; gap: 0.5rem;">
                <input type="hidden" name="page" value="explorer">
                <input type="text" name="category" placeholder="Search or enter category name..." value="<?php echo htmlspecialchars($selectedCategory); ?>">
                <button type="submit" class="btn btn-primary">Search</button>
            </form>

            <h3 style="font-size: 1.1rem; color: #64748b;">Available Categories</h3>
            <div class="grid">
                <?php foreach ($availableCategories as $cat): ?>
                    <a href="?page=explorer&category=<?php echo urlencode($cat); ?>" class="cat-item <?php echo $selectedCategory == $cat ? 'active' : ''; ?>">
                        <?php echo htmlspecialchars($cat); ?>
                    </a>
                <?php endforeach; ?>
            </div>

            <?php if ($selectedCategory && is_dir($baseDir . '/' . $selectedCategory)): ?>
                <h3 style="margin-top: 2.5rem;">Files in "<?php echo htmlspecialchars($selectedCategory); ?>"</h3>
                <ul class="file-list">
                    <?php 
                    $files = array_diff(scandir($baseDir . '/' . $selectedCategory), array('.', '..'));
                    if (empty($files)) echo "<p>No files found in this category.</p>";
                    foreach ($files as $file): 
                        $fileNameOnly = basename($file, ".txt");
                        $countDir = $tmpDir . '/' . $fileNameOnly;
                        $count = is_dir($countDir) ? count(array_diff(scandir($countDir), array('.', '..'))) : 0;
                    ?>
                        <li class="file-item">
                            <a href="?page=explorer&category=<?php echo urlencode($selectedCategory); ?>&view=<?php echo urlencode($file); ?>" class="file-link">
                                <?php echo htmlspecialchars($file); ?>
                            </a>
                            <span class="access-count"><?php echo $count; ?> unique views</span>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>

            <?php if ($viewFile && $selectedCategory): 
                $filePath = $baseDir . '/' . $selectedCategory . '/' . $viewFile;
                if (file_exists($filePath)): ?>
                    <div style="margin-top: 2rem; padding: 1.5rem; background: #f8fafc; border: 1px solid #e2e8f0; border-radius: 0.5rem;">
                        <h4 style="margin-top: 0;">File Content: <span style="font-family: monospace; font-weight: normal;"><?php echo htmlspecialchars($viewFile); ?></span></h4>
                        <pre style="white-space: pre-wrap; background: white; padding: 1rem; border-radius: 0.3rem; border: 1px solid #e2e8f0;"><?php echo htmlspecialchars(file_get_contents($filePath)); ?></pre>
                        <a href="?page=explorer&category=<?php echo urlencode($selectedCategory); ?>" style="font-size: 0.9rem; color: #64748b; text-decoration: none;">&times; Close Preview</a>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <footer style="text-align: center; padding: 3rem 0; color: #94a3b8; font-size: 0.9rem;">
        &copy; <?php echo date('Y'); ?> Rhyzom Project. Secure & Private Content Management.
    </footer>
</div>

</body>
</html>
