<?php
// Define directories
define('UPLOAD_DIR', 'files/');
define('IP_DIR', 'user_ip/');
define('TEXTAREA_DIR', 'textarea_hash/');
define('MAX_FILE_SIZE', 10 * 1024 * 1024); 

// Create directories if they don't exist
foreach ([UPLOAD_DIR, IP_DIR, TEXTAREA_DIR] as $dir) {
    if (!file_exists($dir)) {
        mkdir($dir, 0755, true);
    }
}

$uploadMessage = '';
$uploadSuccess = false;

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['uploaded_file'])) {
    $file = $_FILES['uploaded_file'];
    
    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $uploadMessage = 'Error uploading file.';
    } else {
        // Check file size
        if ($file['size'] > MAX_FILE_SIZE) {
            $uploadMessage = 'File must be smaller than 10mb.';
        } else {
            // Get file extension
            $originalName = $file['name'];
            $extension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
            
            // Check if extension is not php
            if ($extension === 'php') {
                $uploadMessage = 'PHP files are not allowed.';
            } else {
                // Create IP hash
                $userIP = $_SERVER['REMOTE_ADDR'];
                $ipHash = hash('sha256', $userIP);
                $ipFilePath = IP_DIR . $ipHash;
                
                // Check if IP hash file exists
                if (file_exists($ipFilePath)) {
                    $uploadMessage = 'You have already uploaded a file.';
                } else {
                    // Create file hash
                    $fileContent = file_get_contents($file['tmp_name']);
                    $fileHash = hash('sha256', $fileContent);
                    
                    // Create filename with hash and original extension
                    $filename = $fileHash . '.' . $extension;
                    $destination = UPLOAD_DIR . $filename;
                    
                    // Move uploaded file
                    if (move_uploaded_file($file['tmp_name'], $destination)) {
                        // Create IP hash file
                        file_put_contents($ipFilePath, date('Y-m-d H:i:s'));
                        
                        // Save textarea content if provided
                        if (isset($_POST['payment_info']) && !empty(trim($_POST['payment_info']))) {
                            $textareaFile = TEXTAREA_DIR . $fileHash . '.txt';
                            file_put_contents($textareaFile, $_POST['payment_info']);
                        }
                        
                        $uploadMessage = 'File uploaded successfully!';
                        $uploadSuccess = true;
                    } else {
                        $uploadMessage = 'Failed to save file.';
                    }
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Share.</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            background-color: #2ecc71; /* Green background */
            font-family: Arial, sans-serif;
            color: #fff;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            padding: 40px 20px;
            flex: 1;
        }
        
        .main-text {
            text-align: center;
            font-size: 4rem;
            font-weight: bold;
            margin-bottom: 40px;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
        }
        
        .upload-form {
            background-color: rgba(255, 255, 255, 0.1);
            padding: 30px;
            border-radius: 10px;
            backdrop-filter: blur(5px);
            max-width: 500px;
            margin: 0 auto;
        }
        
        .upload-btn {
            display: block;
            width: 200px;
            margin: 0 auto 20px;
            padding: 15px;
            background-color: #000;
            color: #fff;
            border: none;
            border-radius: 5px;
            font-size: 1.2rem;
            cursor: pointer;
            transition: transform 0.2s, background-color 0.2s;
        }
        
        .upload-btn:hover {
            transform: translateY(-2px);
            background-color: #333;
        }
        
        .file-input {
            display: block;
            margin: 0 auto 20px;
            padding: 10px;
            background-color: #fff;
            border-radius: 5px;
            width: 100%;
            max-width: 300px;
        }
        
        .message {
            text-align: center;
            margin: 20px 0;
            padding: 10px;
            border-radius: 5px;
            font-weight: bold;
        }
        
        .success {
            background-color: rgba(46, 204, 113, 0.8);
            color: #fff;
        }
        
        .error {
            background-color: rgba(231, 76, 60, 0.8);
            color: #fff;
        }
        
        textarea {
            width: 100%;
            min-height: 150px;
            padding: 15px;
            border-radius: 5px;
            border: none;
            font-size: 1rem;
            resize: vertical;
            margin-top: 20px;
        }
        
        .textarea-label {
            display: block;
            margin-bottom: 10px;
            font-size: 1.2rem;
        }
        
        .footer {
            text-align: center;
            padding: 20px;
            background-color: rgba(0, 0, 0, 0.2);
            margin-top: 40px;
        }
        
        .footer-links {
            display: flex;
            justify-content: center;
            gap: 30px;
            flex-wrap: wrap;
        }
        
        .footer-link {
            color: #fff;
            text-decoration: none;
            padding: 10px 20px;
            background-color: rgba(0, 0, 0, 0.3);
            border-radius: 5px;
            transition: background-color 0.2s;
        }
        
        .footer-link:hover {
            background-color: rgba(0, 0, 0, 0.5);
        }
        
        .instructions {
            background-color: rgba(255, 255, 255, 0.1);
            padding: 20px;
            border-radius: 10px;
            margin-top: 30px;
            font-size: 0.9rem;
        }
        
        .instructions ul {
            list-style-position: inside;
            margin-top: 10px;
        }
        
        .instructions li {
            margin-bottom: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="main-text">Share.</div>
        
        <?php if ($uploadMessage): ?>
            <div class="message <?php echo $uploadSuccess ? 'success' : 'error'; ?>">
                <?php echo htmlspecialchars($uploadMessage); ?>
            </div>
        <?php endif; ?>
        
        <form method="POST" enctype="multipart/form-data" class="upload-form">
            <input type="file" name="uploaded_file" class="file-input" required>
            <button type="submit" class="upload-btn">Upload</button>
            
            <label class="textarea-label">Payment Information (Optional):</label>
            <textarea name="payment_info" placeholder="Enter your payment information here..."></textarea>
        </form>
        
        <div class="instructions">
            <strong>Upload Requirements:</strong>
            <ul>
                <li>Maximum file size: 50KB</li>
                <li>PHP files are not allowed</li>
                <li>Each IP address can upload only once</li>
                <li>File will be renamed using SHA256 hash + original extension</li>
                <li>Payment information will be saved separately</li>
            </ul>
        </div>
    </div>
    
    <div class="footer">
        <div class="footer-links">
            <a href="#" onclick="downloadSourceCode()" class="footer-link">Download Source Code</a>
            <a href="https://github.com" target="_blank" class="footer-link">Open GitHub (New Tab)</a>
        </div>
    </div>

    <script>
        function downloadSourceCode() {
            // Get the current page content
            fetch(window.location.href)
                .then(response => response.text())
                .then(data => {
                    // Create a blob with the content
                    const blob = new Blob([data], { type: 'text/php' });
                    
                    // Create download link
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = 'share_source.php';
                    
                    // Trigger download
                    document.body.appendChild(a);
                    a.click();
                    
                    // Cleanup
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);
                })
                .catch(error => {
                    console.error('Error downloading source:', error);
                    alert('Error downloading source code. Please try again.');
                });
        }
    </script>
</body>
</html>