<?php
// Create necessary directories if they don't exist
$dirs = ['tmp_url', 'tmp_url_ip', 'tmp_url_server'];
foreach ($dirs as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// Initialize variables
$message = '';
$messageType = '';

// Function to check if URL is online
function isUrlOnline($url) {
    $headers = @get_headers($url);
    return $headers && strpos($headers[0], '200') !== false;
}

// Function to get IP from URL
function getIpFromUrl($url) {
    $host = parse_url($url, PHP_URL_HOST);
    return gethostbyname($host);
}

// Function to check if line exists in file
function lineExistsInFile($file, $line) {
    if (!file_exists($file)) {
        return false;
    }
    $content = file_get_contents($file);
    $lines = explode("\n", trim($content));
    return in_array(trim($line), array_map('trim', $lines));
}

// Function to process URL submission
function processUrl($url, $type, $userIp) {
    global $message, $messageType;
    
    // Validate URL
    if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) {
        $message = 'Invalid URL';
        $messageType = 'error';
        return;
    }
    
    // Validate type
    if ($type !== 'file' && $type !== 'server') {
        $message = 'Invalid type';
        $messageType = 'error';
        return;
    }
    
    // Generate hash for URL
    $urlHash = hash('sha256', $url);
    $hashFile = "tmp_url/{$urlHash}.txt";
    
    // Check if URL hash already exists
    if (file_exists($hashFile)) {
        $message = 'URL already processed';
        $messageType = 'info';
        return;
    }
    
    // Check if URL is online
    if (!isUrlOnline($url)) {
        $message = 'URL is not online';
        $messageType = 'error';
        return;
    }
    
    $targetFile = ($type === 'file') ? 'files.txt' : 'servers.txt';
    
    // Check if URL already exists in target file
    if (lineExistsInFile($targetFile, $url)) {
        $message = 'URL already exists in ' . $targetFile;
        $messageType = 'info';
        return;
    }
    
    // Additional checks for server type
    if ($type === 'server') {
        // Check user IP hash
        $userIpHash = hash('sha256', $userIp);
        $userIpHashFile = "tmp_url_ip/{$userIpHash}.txt";
        
        if (file_exists($userIpHashFile)) {
            $message = 'User IP already used';
            $messageType = 'error';
            return;
        }
        
        // Check URL server IP hash
        $urlIp = getIpFromUrl($url);
        $urlIpHash = hash('sha256', $urlIp);
        $urlIpHashFile = "tmp_url_server/{$urlIpHash}.txt";
        
        if (file_exists($urlIpHashFile)) {
            $message = 'Server IP already used';
            $messageType = 'error';
            return;
        }
        
        // Create user IP hash file
        $fp = fopen($userIpHashFile, 'w');
        fclose($fp);
        
        // Create URL IP hash file
        $fp = fopen($urlIpHashFile, 'w');
        fclose($fp);
    }
    
    // Write URL to target file
    $fp = fopen($targetFile, 'a');
    if ($fp) {
        fwrite($fp, $url . "\n");
        fclose($fp);
        
        // Create URL hash file
        $fp = fopen($hashFile, 'w');
        fclose($fp);
        
        $message = 'URL successfully added to ' . $targetFile;
        $messageType = 'success';
    } else {
        $message = 'Error writing to file';
        $messageType = 'error';
    }
}

// Get user IP
$userIp = $_SERVER['REMOTE_ADDR'];

// Handle GET parameters
if (isset($_GET['file']) && !empty($_GET['file'])) {
    processUrl($_GET['file'], 'file', $userIp);
} elseif (isset($_GET['server']) && !empty($_GET['server'])) {
    processUrl($_GET['server'], 'server', $userIp);
}

// Handle POST submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $url = isset($_POST['url']) ? trim($_POST['url']) : '';
    $type = isset($_POST['type']) ? $_POST['type'] : '';
    
    processUrl($url, $type, $userIp);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>URL Submission</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #f5f5f5;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            width: 100%;
            max-width: 900px;
        }
        
        .form-wrapper {
            background: white;
            padding: 50px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        h1 {
            text-align: center;
            margin-bottom: 40px;
            color: #333;
            font-weight: 300;
            font-size: 32px;
        }
        
        form {
            display: flex;
            flex-direction: column;
            gap: 20px;
        }
        
        .input-group {
            display: flex;
            gap: 15px;
            align-items: stretch;
        }
        
        input[type="text"] {
            flex: 1;
            padding: 20px 25px;
            font-size: 18px;
            border: 2px solid #e0e0e0;
            border-radius: 4px;
            outline: none;
            transition: border-color 0.3s;
            height: 70px;
        }
        
        input[type="text"]:focus {
            border-color: #333;
        }
        
        .radio-group {
            display: flex;
            gap: 20px;
            justify-content: center;
            padding: 10px 0;
        }
        
        .radio-option {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 18px;
            cursor: pointer;
        }
        
        .radio-option input[type="radio"] {
            width: 20px;
            height: 20px;
            cursor: pointer;
        }
        
        button {
            padding: 20px 40px;
            font-size: 18px;
            background: #333;
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            transition: background 0.3s;
            height: 70px;
            min-width: 150px;
        }
        
        button:hover {
            background: #555;
        }
        
        .message {
            padding: 20px;
            border-radius: 4px;
            margin-bottom: 20px;
            text-align: center;
            font-size: 16px;
        }
        
        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .message.info {
            background: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        
        @media (max-width: 768px) {
            .form-wrapper {
                padding: 30px 20px;
            }
            
            .input-group {
                flex-direction: column;
            }
            
            h1 {
                font-size: 24px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="form-wrapper">
            <h1>URL Submission</h1>
            
            <?php if ($message): ?>
                <div class="message <?php echo $messageType; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <form method="POST">
                <input 
                    type="text" 
                    name="url" 
                    placeholder="Enter URL..." 
                    required
                    value="<?php echo isset($_POST['url']) ? htmlspecialchars($_POST['url']) : ''; ?>"
                >
                
                <div class="radio-group">
                    <label class="radio-option">
                        <input 
                            type="radio" 
                            name="type" 
                            value="file" 
                            required
                            <?php echo (!isset($_POST['type']) || $_POST['type'] === 'file') ? 'checked' : ''; ?>
                        >
                        <span>File</span>
                    </label>
                    <label class="radio-option">
                        <input 
                            type="radio" 
                            name="type" 
                            value="server"
                            <?php echo (isset($_POST['type']) && $_POST['type'] === 'server') ? 'checked' : ''; ?>
                        >
                        <span>Server</span>
                    </label>
                </div>
                
                <div class="input-group">
                    <button type="submit">Submit</button>
                </div>
            </form>
        </div>
    </div>
</body>
</html>