<?php
session_start();

// Initialize session variable if not set
if (!isset($_SESSION['user'])) {
    $_SESSION['user'] = 'default_user_' . time();
}

// Configuration
$salt = 'your_salt_here'; // Change this to a secure salt
$rizoma_dir = 'rizoma/';
$files_dir = 'files/';
$file_hash_dir = 'file_hash/';
$url_hash_dir = 'url_hash/';
$max_file_size = 200 * 1024 * 1024; // 200MB in bytes

// Create directories if they don't exist
foreach ([$rizoma_dir, $files_dir, $file_hash_dir, $url_hash_dir] as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// Function to check if URL is valid
function isValidUrl($url) {
    return filter_var($url, FILTER_VALIDATE_URL) !== false;
}

// Function to check if URL is online
function isUrlOnline($url) {
    $headers = @get_headers($url);
    return $headers && strpos($headers[0], '200') !== false;
}

// Function to get file extension
function getFileExtension($url) {
    $path = parse_url($url, PHP_URL_PATH);
    return pathinfo($path, PATHINFO_EXTENSION);
}

// Function to get file content with curl
function downloadFile($url, $max_size) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_HEADER, false);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_BUFFERSIZE, 128);
    curl_setopt($ch, CURLOPT_NOPROGRESS, false);
    
    // Limit download size
    $downloaded = 0;
    curl_setopt($ch, CURLOPT_PROGRESSFUNCTION, function($resource, $download_size, $downloaded_size, $upload_size, $uploaded_size) use ($max_size, &$downloaded) {
        $downloaded = $downloaded_size;
        return ($downloaded_size > $max_size) ? 1 : 0;
    });
    
    $content = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return ($http_code == 200 && strlen($content) <= $max_size) ? $content : false;
}

// Function to write file only if it doesn't exist
function writeFileIfNotExists($filename, $content) {
    if (!file_exists($filename)) {
        return file_put_contents($filename, $content) !== false;
    }
    return false;
}

// Process form submission
$results = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['urls'])) {
    $user_ip = $_SERVER['REMOTE_ADDR'];
    $ip_hash = hash('sha256', $user_ip . $salt);
    $rizoma_file = $rizoma_dir . $ip_hash . '.txt';
    
    $urls = explode("\n", trim($_POST['urls']));
    $valid_urls = [];
    
    foreach ($urls as $url) {
        $url = trim($url);
        if (empty($url)) continue;
        
        $url_hash = hash('sha256', $url);
        $url_hash_file = $url_hash_dir . $url_hash;
        
        // Skip if URL hash already exists
        if (file_exists($url_hash_file)) {
            $results[] = "Skipped (already processed): $url";
            continue;
        }
        
        // Validate URL
        if (!isValidUrl($url)) {
            $results[] = "Invalid URL: $url";
            continue;
        }
        
        // Check if URL is online
        if (!isUrlOnline($url)) {
            $results[] = "URL offline: $url";
            continue;
        }
        
        // Create URL hash file
        if (writeFileIfNotExists($url_hash_file, '')) {
            $results[] = "URL hash created: $url";
            
            // Add to valid URLs for rizoma file
            $valid_urls[] = $url;
            
            // Try to download file if it's smaller than 200MB
            $file_content = downloadFile($url, $max_file_size);
            
            if ($file_content !== false) {
                $file_hash = hash('sha256', $file_content);
                $extension = getFileExtension($url);
                $filename = $file_hash . ($extension ? '.' . $extension : '');
                $file_path = $files_dir . $filename;
                
                // Save file if it doesn't exist
                if (writeFileIfNotExists($file_path, $file_content)) {
                    $results[] = "File downloaded: $filename";
                    
                    // Create file hash file if it doesn't exist
                    $file_hash_path = $file_hash_dir . $file_hash;
                    if (writeFileIfNotExists($file_hash_path, $_SESSION['user'])) {
                        $results[] = "File hash created for: $filename";
                    }
                } else {
                    $results[] = "File already exists: $filename";
                }
            } else {
                $results[] = "File too large or download failed: $url";
            }
        }
    }
    
    // Save valid URLs to rizoma file (overwrite if exists)
    if (!empty($valid_urls)) {
        file_put_contents($rizoma_file, implode("\n", $valid_urls));
        $results[] = "Rizoma file saved/updated: $rizoma_file";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>URL Processor</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        .container {
            background: #f5f5f5;
            padding: 20px;
            border-radius: 5px;
        }
        textarea {
            width: 100%;
            height: 200px;
            padding: 10px;
            margin-bottom: 10px;
            border: 1px solid #ddd;
            border-radius: 3px;
        }
        button {
            background: #007bff;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 3px;
            cursor: pointer;
        }
        button:hover {
            background: #0056b3;
        }
        .results {
            margin-top: 20px;
            padding: 15px;
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 3px;
        }
        .result-item {
            padding: 5px 0;
            border-bottom: 1px solid #eee;
        }
        .result-item:last-child {
            border-bottom: none;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>RIZOMADB</h1>
        <form method="POST">
            <textarea name="urls" placeholder="Enter URLs (one per line)..." required><?php echo isset($_POST['urls']) ? htmlspecialchars($_POST['urls']) : ''; ?></textarea>
            <button type="submit">Download</button>
        </form>
        
        <?php if (!empty($results)): ?>
        <div class="results">
            <h3>Results:</h3>
            <?php foreach ($results as $result): ?>
                <div class="result-item"><?php echo htmlspecialchars($result); ?></div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
        
        <div style="margin-top: 20px; font-size: 0.9em; color: #666;">
            <p><strong>Session User:</strong> <?php echo htmlspecialchars($_SESSION['user']); ?></p>
            <p><strong>Your IP:</strong> <?php echo htmlspecialchars($_SERVER['REMOTE_ADDR']); ?></p>
        </div>
    </div>
</body>
</html>