<?php
// Configuration
$baseDir = 'rhyzom';
$message = "";
$savedCount = 0;

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // 1. Process Categories (Max 5, max 12 chars, Alphanumeric + _)
    $categories = [];
    if (isset($_POST['categories']) && is_array($_POST['categories'])) {
        foreach (array_slice($_POST['categories'], 0, 5) as $cat) {
            $cat = trim($cat);
            // Validation: alphanumeric + underscore, max 12 chars
            if (preg_match('/^[a-zA-Z0-9_]{1,12}$/', $cat)) {
                $categories[] = $cat;
            }
        }
    }

    // 2. Process Signature (Max 50 chars) -> SHA-256 Hash
    $signature = substr(trim($_POST['signature'] ?? ''), 0, 50);
    $fileHash = hash('sha256', $signature);

    // 3. Process Textarea (Line 1: Metadata, Lines 2+: Valid URLs)
    $contentRaw = $_POST['content'] ?? '';
    $lines = explode("\n", str_replace("\r", "", $contentRaw));
    $metadata = $lines[0] ?? '';
    $validUrls = [];

    for ($i = 1; $i < count($lines); $i++) {
        $url = trim($lines[$i]);
        if (!empty($url) && filter_var($url, FILTER_VALIDATE_URL)) {
            $validUrls[] = $url;
        }
    }

    // 4. File Writing Logic (Iterating through each category)
    if (empty($categories)) {
        $message = "Error: At least one valid category is required.";
    } elseif (empty($signature)) {
        $message = "Error: Signature is required.";
    } else {
        // Prepare file content once
        $fileData = "" . $metadata . "\n";
        $fileData .= "" . implode("\n", $validUrls);

        foreach ($categories as $folderName) {
            $targetFolder = $baseDir . DIRECTORY_SEPARATOR . $folderName;
            
            // Create directory if it doesn't exist
            if (!is_dir($targetFolder)) {
                mkdir($targetFolder, 0755, true);
            }

            $filePath = $targetFolder . DIRECTORY_SEPARATOR . $fileHash . ".txt";
            
            // Write using fopen ("w" overwrites existing files)
            $handle = fopen($filePath, "w");
            if ($handle) {
                fwrite($handle, $fileData);
                fclose($handle);
                $savedCount++;
            }
        }
        $message = "Success: Content saved in $savedCount category folders.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Multi-Category Data Portal</title>
    <style>
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background: #eef2f3; padding: 40px 20px; }
        .container { max-width: 650px; margin: 0 auto; background: #fff; padding: 30px; border-radius: 12px; box-shadow: 0 4px 15px rgba(0,0,0,0.1); }
        h2 { border-bottom: 2px solid #eee; padding-bottom: 10px; color: #2c3e50; }
        .field-group { margin-bottom: 20px; }
        label { display: block; font-weight: 600; margin-bottom: 8px; color: #34495e; }
        input[type="text"], textarea { width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 6px; box-sizing: border-box; font-size: 14px; }
        textarea { height: 120px; font-family: 'Courier New', monospace; resize: vertical; }
        .cat-inputs { display: grid; grid-template-columns: repeat(auto-fit, minmax(110px, 1fr)); gap: 10px; }
        button { width: 100%; background: #27ae60; color: white; border: none; padding: 14px; border-radius: 6px; font-size: 16px; font-weight: bold; cursor: pointer; transition: background 0.3s; }
        button:hover { background: #219150; }
        .message { padding: 15px; margin-bottom: 20px; border-radius: 6px; font-weight: 500; }
        .error { background: #fdeaea; color: #c0392b; border: 1px solid #f5c6cb; }
        .success { background: #eafaf1; color: #27ae60; border: 1px solid #d4edda; }
        small { color: #7f8c8d; }
    </style>
</head>
<body>

<div class="container">
    <h2>Data Distribution</h2>
    
    <?php if ($message): ?>
        <div class="message <?php echo strpos($message, 'Error') === 0 ? 'error' : 'success'; ?>">
            <?php echo $message; ?>
        </div>
    <?php endif; ?>

    <form method="POST">
        <div class="field-group">
            <label>Categories <small>(Max 5, Alphanumeric + _ , Max 12 chars)</small></label>
            <div class="cat-inputs">
                <input type="text" name="categories[]" placeholder="Cat 1" maxlength="12" pattern="[A-Za-z0-9_]+">
                <input type="text" name="categories[]" placeholder="Cat 2" maxlength="12" pattern="[A-Za-z0-9_]+">
                <input type="text" name="categories[]" placeholder="Cat 3" maxlength="12" pattern="[A-Za-z0-9_]+">
                <input type="text" name="categories[]" placeholder="Cat 4" maxlength="12" pattern="[A-Za-z0-9_]+">
                <input type="text" name="categories[]" placeholder="Cat 5" maxlength="12" pattern="[A-Za-z0-9_]+">
            </div>
        </div>

        <div class="field-group">
            <label>Content</label>
            <textarea name="content" placeholder="Line 1: BTC/Metadata&#10;Line 2: http://url1.com&#10;Line 3: http://url2.com"></textarea>
        </div>

        <div class="field-group">
            <label>Signature <small>(Max 50 chars - used for SHA-256 filename)</small></label>
            <input type="text" name="signature" maxlength="50" placeholder="Enter signature..." required>
        </div>

        <button type="submit">Distribute to Folders</button>
    </form>
</div>

</body>
</html>