﻿<?php
/**
 * Rhyzom Project - Landing Page, Data Processor & File Explorer
 * All-in-one file: Logic, Landing Page, App, and Explorer.
 */

$page = isset($_GET['page']) ? $_GET['page'] : 'landing';
$message = "";
$messageType = "";
$baseDir = "rhyzom";
$tmpDir = "tmp_rhyzom";

// --- LOGIC SECTION: DATA PROCESSING ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && $page == 'app') {
    try {
        $rawCategories = isset($_POST['categories']) ? $_POST['categories'] : '';
        $categoriesArray = array_filter(array_map('trim', explode(',', $rawCategories)));
        
        if (count($categoriesArray) > 5) throw new Exception("Error: Max 5 categories.");
        foreach ($categoriesArray as $cat) {
            if (strlen($cat) > 12 || !preg_match('/^[a-zA-Z0-9_]+$/', $cat)) {
                throw new Exception("Error: Invalid category '$cat'.");
            }
        }

        $signature = isset($_POST['signature']) ? trim($_POST['signature']) : '';
        if (strlen($signature) > 50 || empty($signature)) throw new Exception("Error: Invalid signature.");
        $signatureHash = hash('sha256', $signature);

        $contentLines = isset($_POST['content']) ? explode("\n", str_replace("\r", "", $_POST['content'])) : [];
        $finalContent = "";
        if (!empty($contentLines)) {
            $finalContent .= $contentLines[0] . "\n";
            for ($i = 1; $i < count($contentLines); $i++) {
                $line = trim($contentLines[$i]);
                if (!empty($line)) {
                    if (filter_var($line, FILTER_VALIDATE_URL)) $finalContent .= $line . "\n";
                    else throw new Exception("Error: Invalid URL at line " . ($i + 1));
                }
            }
        }

        if (!is_dir($baseDir)) mkdir($baseDir, 0777, true);
        foreach ($categoriesArray as $category) {
            $catPath = $baseDir . DIRECTORY_SEPARATOR . $category;
            if (!is_dir($catPath)) mkdir($catPath, 0777, true);
            $handle = fopen($catPath . DIRECTORY_SEPARATOR . $signatureHash . ".txt", "w");
            if ($handle) { fwrite($handle, $finalContent); fclose($handle); }
        }
        $message = "Success! Files saved.";
        $messageType = "success";
    } catch (Exception $e) {
        $message = $e->getMessage();
        $messageType = "error";
    }
}

// --- LOGIC SECTION: FILE EXPLORER & ACCESS COUNTING ---
$selectedCategory = isset($_GET['category']) ? $_GET['category'] : '';
$viewFile = isset($_GET['view']) ? $_GET['view'] : '';

if ($viewFile && $selectedCategory) {
    // Access Counting Logic
    $userIp = $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    $ipHash = hash('sha256', $userIp);
    $fileNameOnly = basename($viewFile, ".txt");
    $accessLogDir = $tmpDir . DIRECTORY_SEPARATOR . $fileNameOnly;
    
    if (!is_dir($accessLogDir)) mkdir($accessLogDir, 0777, true);
    $logFile = $accessLogDir . DIRECTORY_SEPARATOR . $ipHash;
    $handle = fopen($logFile, "w"); // Create empty file with IP hash as name
    if ($handle) fclose($handle);
}

// Helper to get categories
$availableCategories = is_dir($baseDir) ? array_diff(scandir($baseDir), array('.', '..')) : [];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rhyzom - Secure Data System</title>
    <style>
        :root { --primary: #2563eb; --primary-hover: #1d4ed8; --success: #059669; --error: #dc2626; --bg: #f8fafc; --text: #1e293b; --card: #ffffff; }
        body { font-family: 'Inter', sans-serif; background: var(--bg); color: var(--text); margin: 0; line-height: 1.6; }
        .container { max-width: 1000px; margin: 0 auto; padding: 2rem; }
        nav { display: flex; justify-content: space-between; align-items: center; padding: 1rem 0; border-bottom: 1px solid #e2e8f0; margin-bottom: 2rem; }
        .logo { font-size: 1.5rem; font-weight: 800; color: var(--primary); text-decoration: none; }
        .nav-links a { text-decoration: none; color: var(--text); font-weight: 500; margin-left: 1.5rem; }
        .card { background: var(--card); padding: 2rem; border-radius: 1rem; box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1); margin-bottom: 2rem; }
        .btn { display: inline-block; padding: 0.75rem 1.5rem; border-radius: 0.5rem; text-decoration: none; font-weight: 600; cursor: pointer; border: none; }
        .btn-primary { background: var(--primary); color: white; }
        .btn-success { background: var(--success); color: white; width: 100%; }
        .form-group { margin-bottom: 1.2rem; }
        label { display: block; margin-bottom: 0.4rem; font-weight: 600; }
        input, textarea { width: 100%; padding: 0.75rem; border: 1px solid #cbd5e1; border-radius: 0.375rem; box-sizing: border-box; }
        .grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; }
        .cat-item { padding: 1rem; background: #f1f5f9; border-radius: 0.5rem; text-align: center; text-decoration: none; color: var(--text); font-weight: 600; transition: background 0.2s; }
        .cat-item:hover, .cat-item.active { background: var(--primary); color: white; }
        .file-list { list-style: none; padding: 0; }
        .file-item { display: flex; justify-content: space-between; align-items: center; padding: 0.75rem; border-bottom: 1px solid #f1f5f9; }
        .file-link { color: var(--primary); text-decoration: none; font-family: monospace; font-size: 0.9rem; }
        .alert { padding: 1rem; border-radius: 0.375rem; margin-bottom: 1.5rem; }
        .alert-success { background: #dcfce7; color: #166534; }
        .alert-error { background: #fee2e2; color: #991b1b; }
        .access-count { font-size: 0.8rem; color: #64748b; background: #e2e8f0; padding: 0.2rem 0.5rem; border-radius: 1rem; }
    </style>
</head>
<body>

<div class="container">
    <nav>
        <a href="?page=landing" class="logo">RHYZOM</a>
        <div class="nav-links">
            <a href="?page=landing">Home</a>
            <a href="?page=app">Processor</a>
            <a href="?page=explorer">Explorer</a>
        </div>
    </nav>

    <?php if ($page == 'landing'): ?>
        <section style="text-align: center; padding: 3rem 0;">
            <h1>Secure Data Ecosystem</h1>
            <p style="font-size: 1.2rem; color: #64748b;">Categorize, Sign, and Track your data with cryptographic precision.</p>
            <div style="margin-top: 2rem;">
                <a href="?page=app" class="btn btn-primary">Start Processing</a>
                <a href="?page=explorer" class="btn" style="background: #e2e8f0; margin-left: 1rem;">Explore Files</a>
            </div>
        </section>

    <?php elseif ($page == 'app'): ?>
        <div class="card">
            <h2>Data Processor</h2>
            <?php if ($message): ?><div class="alert alert-<?php echo $messageType; ?>"><?php echo $message; ?></div><?php endif; ?>
            <form method="POST" action="?page=app">
                <div class="form-group"><label>Categories (max 5, comma separated)</label><input type="text" name="categories" required></div>
                <div class="form-group"><label>Signature (max 50 chars)</label><input type="text" name="signature" required></div>
                <div class="form-group"><label>Content (Line 1: Meta, Line 2+: URLs)</label><textarea name="content" required></textarea></div>
                <button type="submit" class="btn btn-success">Process & Save</button>
            </form>
        </div>

    <?php elseif ($page == 'explorer'): ?>
        <div class="card">
            <h2>File Explorer</h2>
            <form method="GET" action="" style="margin-bottom: 2rem; display: flex; gap: 0.5rem;">
                <input type="hidden" name="page" value="explorer">
                <input type="text" name="category" placeholder="Enter category name to search..." value="<?php echo htmlspecialchars($selectedCategory); ?>">
                <button type="submit" class="btn btn-primary">Search</button>
            </form>

            <h3>Categories</h3>
            <div class="grid">
                <?php foreach ($availableCategories as $cat): ?>
                    <a href="?page=explorer&category=<?php echo urlencode($cat); ?>" class="cat-item <?php echo $selectedCategory == $cat ? 'active' : ''; ?>">
                        <?php echo htmlspecialchars($cat); ?>
                    </a>
                <?php endforeach; ?>
            </div>

            <?php if ($selectedCategory && is_dir($baseDir . '/' . $selectedCategory)): ?>
                <h3 style="margin-top: 2rem;">Files in "<?php echo htmlspecialchars($selectedCategory); ?>"</h3>
                <ul class="file-list">
                    <?php 
                    $files = array_diff(scandir($baseDir . '/' . $selectedCategory), array('.', '..'));
                    foreach ($files as $file): 
                        $fileNameOnly = basename($file, ".txt");
                        $countDir = $tmpDir . '/' . $fileNameOnly;
                        $count = is_dir($countDir) ? count(array_diff(scandir($countDir), array('.', '..'))) : 0;
                    ?>
                        <li class="file-item">
                            <a href="?page=explorer&category=<?php echo urlencode($selectedCategory); ?>&view=<?php echo urlencode($file); ?>" class="file-link">
                                <?php echo htmlspecialchars($file); ?>
                            </a>
                            <span class="access-count"><?php echo $count; ?> unique views</span>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>

            <?php if ($viewFile && $selectedCategory): 
                $filePath = $baseDir . '/' . $selectedCategory . '/' . $viewFile;
                if (file_exists($filePath)): ?>
                    <div style="margin-top: 2rem; padding: 1.5rem; background: #f8fafc; border: 1px solid #e2e8f0; border-radius: 0.5rem;">
                        <h4>File Content: <?php echo htmlspecialchars($viewFile); ?></h4>
                        <pre style="white-space: pre-wrap;"><?php echo htmlspecialchars(file_get_contents($filePath)); ?></pre>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <footer style="text-align: center; padding: 2rem; color: #94a3b8;">
        &copy; <?php echo date('Y'); ?> Rhyzom Project.
    </footer>
</div>

</body>
</html>
