﻿<?php
// Start session for form persistence
session_start();

// Configuration
$MAX_CATEGORIES = 5;
$CATEGORY_MAX_LENGTH = 12;
$SIGNATURE_MAX_LENGTH = 50;
$STORE_DIR = "rhyzom";

// Ensure the storage directory exists
if (!file_exists($STORE_DIR) && !is_dir($STORE_DIR)) {
    mkdir($STORE_DIR, 0755, true);
}

// Initialize form data from session or defaults
$categories = $_SESSION['categories'] ?? ['', '', '', '', ''];
$links = $_SESSION['links'] ?? '';
$signature = $_SESSION['signature'] ?? '';
$message = '';
$message_type = ''; // 'success' or 'error'

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Reset session data
    $_SESSION['categories'] = [];
    $_SESSION['links'] = '';
    $_SESSION['signature'] = '';
    
    // Validate and process categories
    $valid_categories = [];
    $category_errors = [];
    
    for ($i = 0; $i < $MAX_CATEGORIES; $i++) {
        $cat_key = 'category_' . $i;
        $category = isset($_POST[$cat_key]) ? trim($_POST[$cat_key]) : '';
        
        if (!empty($category)) {
            // Check length
            if (strlen($category) > $CATEGORY_MAX_LENGTH) {
                $category_errors[] = "Category '$category' exceeds maximum length of $CATEGORY_MAX_LENGTH characters.";
            }
            
            // Check format (only letters, numbers, and underscore)
            if (!preg_match('/^[a-zA-Z0-9_]+$/', $category)) {
                $category_errors[] = "Category '$category' contains invalid characters. Only letters, numbers, and underscore are allowed.";
            }
            
            $valid_categories[] = $category;
        } else {
            $valid_categories[] = '';
        }
        
        // Store in session
        $_SESSION['categories'][$i] = $category;
    }
    
    // Process links
    $links_text = isset($_POST['links']) ? trim($_POST['links']) : '';
    $_SESSION['links'] = $links_text;
    
    $links_array = explode("\n", $links_text);
    $link_errors = [];
    
    // Validate each link (skip first line)
    for ($i = 1; $i < count($links_array); $i++) {
        $link = trim($links_array[$i]);
        if (!empty($link)) {
            // Validate URL format
            if (!filter_var($link, FILTER_VALIDATE_URL)) {
                $line_num = $i + 1;
                $link_errors[] = "Line $line_num: '$link' is not a valid URL.";
            }
        }
    }
    
    // Process signature
    $signature = isset($_POST['signature']) ? trim($_POST['signature']) : '';
    $_SESSION['signature'] = $signature;
    
    // Validate signature length
    if (!empty($signature) && strlen($signature) > $SIGNATURE_MAX_LENGTH) {
        $message = "Signature exceeds maximum length of $SIGNATURE_MAX_LENGTH characters.";
        $message_type = 'error';
    }
    
    // If there are errors, display them
    if (!empty($category_errors) || !empty($link_errors)) {
        $message = "Please correct the following errors:<br>";
        foreach (array_merge($category_errors, $link_errors) as $error) {
            $message .= "- $error<br>";
        }
        $message_type = 'error';
    } 
    // If everything is valid, save the file
    else if (!empty($signature) && !empty(array_filter($valid_categories))) {
        // Create hash from signature
        $hash = hash('sha256', $signature);
        
        // Get the first non-empty category
        $selected_category = '';
        foreach ($valid_categories as $cat) {
            if (!empty($cat)) {
                $selected_category = $cat;
                break;
            }
        }
        
        // Ensure category directory exists
        $category_dir = $STORE_DIR . '/' . $selected_category;
        if (!file_exists($category_dir) && !is_dir($category_dir)) {
            mkdir($category_dir, 0755, true);
        }
        
        // Prepare file content
        $content .= "$signature\n";
        $content .= $links_text;
        
        // Save to file (overwrite if exists)
        $filename = $category_dir . '/' . $hash . '.txt';
        $file = fopen($filename, 'w');
        
        if ($file) {
            fwrite($file, $content);
            fclose($file);
            
            $message = "File saved successfully to: $filename";
            $message_type = 'success';
            
            // Clear form after successful save
            $_SESSION['categories'] = ['', '', '', '', ''];
            $_SESSION['links'] = '';
            $_SESSION['signature'] = '';
            
            // Update local variables for display
            $categories = ['', '', '', '', ''];
            $links = '';
            $signature = '';
        } else {
            $message = "Error: Could not save the file.";
            $message_type = 'error';
        }
    } else if (empty($signature)) {
        $message = "Please provide a signature.";
        $message_type = 'error';
    } else if (empty(array_filter($valid_categories))) {
        $message = "Please provide at least one category.";
        $message_type = 'error';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Category & Link Manager</title>
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        body {
            background-color: #f5f7fa;
            color: #333;
            line-height: 1.6;
            padding: 20px;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: white;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            padding: 30px;
        }
        
        h1 {
            color: #2c3e50;
            margin-bottom: 10px;
            border-bottom: 2px solid #3498db;
            padding-bottom: 10px;
        }
        
        .subtitle {
            color: #7f8c8d;
            margin-bottom: 30px;
        }
        
        .form-section {
            margin-bottom: 25px;
            padding: 20px;
            border-radius: 8px;
            background-color: #f8f9fa;
        }
        
        .section-title {
            color: #3498db;
            margin-bottom: 15px;
            font-size: 1.2em;
            display: flex;
            align-items: center;
        }
        
        .section-title i {
            margin-right: 10px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #2c3e50;
        }
        
        .input-hint {
            font-size: 0.9em;
            color: #7f8c8d;
            margin-bottom: 10px;
        }
        
        input[type="text"], textarea {
            width: 100%;
            padding: 12px;
            border: 2px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        input[type="text"]:focus, textarea:focus {
            border-color: #3498db;
            outline: none;
        }
        
        .category-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            gap: 15px;
        }
        
        .category-input {
            position: relative;
        }
        
        .char-count {
            position: absolute;
            right: 10px;
            top: 10px;
            font-size: 0.8em;
            color: #7f8c8d;
        }
        
        textarea {
            min-height: 150px;
            resize: vertical;
            font-family: monospace;
        }
        
        .button-group {
            display: flex;
            justify-content: space-between;
            margin-top: 30px;
        }
        
        button {
            padding: 12px 25px;
            border: none;
            border-radius: 5px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: background-color 0.3s, transform 0.2s;
        }
        
        button:hover {
            transform: translateY(-2px);
        }
        
        button:active {
            transform: translateY(0);
        }
        
        .submit-btn {
            background-color: #2ecc71;
            color: white;
        }
        
        .submit-btn:hover {
            background-color: #27ae60;
        }
        
        .reset-btn {
            background-color: #e74c3c;
            color: white;
        }
        
        .reset-btn:hover {
            background-color: #c0392b;
        }
        
        .message {
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            display: none;
        }
        
        .success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
            display: block;
        }
        
        .error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
            display: block;
        }
        
        .info-box {
            background-color: #e8f4fc;
            border-left: 4px solid #3498db;
            padding: 15px;
            margin-top: 20px;
            border-radius: 0 5px 5px 0;
        }
        
        .info-title {
            font-weight: bold;
            color: #2c3e50;
            margin-bottom: 5px;
        }
        
        .info-text {
            font-size: 0.95em;
        }
        
        @media (max-width: 600px) {
            .container {
                padding: 20px;
            }
            
            .category-grid {
                grid-template-columns: 1fr;
            }
            
            .button-group {
                flex-direction: column;
                gap: 10px;
            }
            
            button {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Category & Link Manager</h1>
        <p class="subtitle">Store your categories, links, and signature with SHA-256 hashing</p>
        
        <?php if (!empty($message)): ?>
            <div class="message <?php echo $message_type; ?>">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>
        
        <form method="POST" action="">
            <!-- Categories Section -->
            <div class="form-section">
                <div class="section-title">
                    <span>📁</span>
                    <span>Categories (Max 5)</span>
                </div>
                <p class="input-hint">Enter up to 5 categories. Each can contain letters, numbers, and underscores only. Max length: 12 characters.</p>
                
                <div class="category-grid">
                    <?php for ($i = 0; $i < $MAX_CATEGORIES; $i++): ?>
                    <div class="category-input">
                        <input 
                            type="text" 
                            name="category_<?php echo $i; ?>" 
                            id="category_<?php echo $i; ?>" 
                            value="<?php echo htmlspecialchars($categories[$i] ?? ''); ?>" 
                            placeholder="Category <?php echo $i+1; ?>"
                            maxlength="<?php echo $CATEGORY_MAX_LENGTH; ?>"
                            oninput="updateCharCount(this)"
                        >
                        <span class="char-count" id="char-count-<?php echo $i; ?>">
                            <?php echo strlen($categories[$i] ?? ''); ?>/<?php echo $CATEGORY_MAX_LENGTH; ?>
                        </span>
                    </div>
                    <?php endfor; ?>
                </div>
            </div>
            
            <!-- Links Section -->
            <div class="form-section">
                <div class="section-title">
                    <span>🔗</span>
                    <span>Links</span>
                </div>
                <p class="input-hint">Enter links (one per line). First line can be BTC address or metadata. Subsequent lines must be valid URLs.</p>
                
                <label for="links">Links (one per line):</label>
                <textarea 
                    name="links" 
                    id="links" 
                    placeholder="e.g. 1A1zP1eP5QGefi2DMPTfTL5SLmv7DivfNa&#10;https://example.com/link1&#10;https://example.com/link2"
                ><?php echo htmlspecialchars($links); ?></textarea>
            </div>
            
            <!-- Signature Section -->
            <div class="form-section">
                <div class="section-title">
                    <span>✍️</span>
                    <span>Signature</span>
                </div>
                <p class="input-hint">Enter a signature (max 50 characters). It will be hashed with SHA-256 and used as the filename.</p>
                
                <label for="signature">Signature:</label>
                <input 
                    type="text" 
                    name="signature" 
                    id="signature" 
                    value="<?php echo htmlspecialchars($signature); ?>" 
                    placeholder="Enter your signature"
                    maxlength="<?php echo $SIGNATURE_MAX_LENGTH; ?>"
                    oninput="updateSignatureCount(this)"
                >
                <div class="char-count" id="signature-char-count">
                    <?php echo strlen($signature); ?>/<?php echo $SIGNATURE_MAX_LENGTH; ?>
                </div>
            </div>
            
            <!-- Info Box -->
            <div class="info-box">
                <div class="info-title">How it works:</div>
                <div class="info-text">
                    <p>1. Your signature will be converted to SHA-256 hash</p>
                    <p>2. A file with this hash as the name will be created in the "rhyzom/[category]" folder</p>
                    <p>3. The first non-empty category will be used as the folder name</p>
                    <p>4. If a file with the same hash exists, it will be overwritten</p>
                </div>
            </div>
            
            <!-- Buttons -->
            <div class="button-group">
                <button type="button" class="reset-btn" onclick="resetForm()">Reset Form</button>
                <button type="submit" class="submit-btn">Save Data</button>
            </div>
        </form>
    </div>
    
    <script>
        // Update character count for category inputs
        function updateCharCount(input) {
            const id = input.id.split('_')[1];
            const charCount = document.getElementById(`char-count-${id}`);
            charCount.textContent = `${input.value.length}/<?php echo $CATEGORY_MAX_LENGTH; ?>`;
            
            // Validate input (only letters, numbers, underscore)
            const isValid = /^[a-zA-Z0-9_]*$/.test(input.value);
            if (!isValid) {
                input.style.borderColor = '#e74c3c';
                charCount.style.color = '#e74c3c';
            } else {
                input.style.borderColor = '#ddd';
                charCount.style.color = '#7f8c8d';
            }
        }
        
        // Update signature character count
        function updateSignatureCount(input) {
            const charCount = document.getElementById('signature-char-count');
            charCount.textContent = `${input.value.length}/<?php echo $SIGNATURE_MAX_LENGTH; ?>`;
        }
        
        // Reset form function
        function resetForm() {
            if (confirm('Are you sure you want to reset the form? All entered data will be lost.')) {
                // Clear all category inputs
                for (let i = 0; i < 5; i++) {
                    const input = document.getElementById(`category_${i}`);
                    if (input) {
                        input.value = '';
                        updateCharCount(input);
                    }
                }
                
                // Clear links textarea
                const linksTextarea = document.getElementById('links');
                if (linksTextarea) linksTextarea.value = '';
                
                // Clear signature input
                const signatureInput = document.getElementById('signature');
                if (signatureInput) {
                    signatureInput.value = '';
                    updateSignatureCount(signatureInput);
                }
            }
        }
        
        // Initialize character counts on page load
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize category character counts
            for (let i = 0; i < 5; i++) {
                const input = document.getElementById(`category_${i}`);
                if (input) updateCharCount(input);
            }
            
            // Initialize signature character count
            const signatureInput = document.getElementById('signature');
            if (signatureInput) updateSignatureCount(signatureInput);
        });
    </script>
</body>
</html>