<?php
// Configuration
$baseDir = 'rhyzom';
$message = "";
$savedCount = 0;
$currentView = isset($_GET['view']) ? $_GET['view'] : 'landing';

// --- Processing Logic ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $currentView = 'app'; // Keep user on the app view after submission
    
    // 1. Process Categories (Max 5, max 12 chars, Alphanumeric + _)
    $categories = [];
    if (isset($_POST['categories']) && is_array($_POST['categories'])) {
        foreach (array_slice($_POST['categories'], 0, 5) as $cat) {
            $cat = trim($cat);
            if (preg_match('/^[a-zA-Z0-9_]{1,12}$/', $cat) && !empty($cat)) {
                $categories[] = $cat;
            }
        }
    }

    // 2. Process Signature (Max 50 chars) -> SHA-256 Hash
    $signature = substr(trim($_POST['signature'] ?? ''), 0, 50);
    $fileHash = hash('sha256', $signature);

    // 3. Process Textarea (Line 1: Metadata, Lines 2+: Valid URLs)
    $contentRaw = $_POST['content'] ?? '';
    $lines = explode("\n", str_replace("\r", "", $contentRaw));
    $metadata = $lines[0] ?? '';
    $validUrls = [];

    for ($i = 1; $i < count($lines); $i++) {
        $url = trim($lines[$i]);
        if (!empty($url) && filter_var($url, FILTER_VALIDATE_URL)) {
            $validUrls[] = $url;
        }
    }

    // 4. File Writing Logic
    if (empty($categories)) {
        $message = "Error: At least one valid category (alphanumeric, max 12 chars) is required.";
    } elseif (empty($signature)) {
        $message = "Error: Signature is required to generate the file hash.";
    } else {
        $fileData = "Metadata: " . $metadata . "\n";
        $fileData .= "URLs:\n" . implode("\n", $validUrls);

        foreach ($categories as $folderName) {
            $targetFolder = $baseDir . DIRECTORY_SEPARATOR . $folderName;
            if (!is_dir($targetFolder)) {
                mkdir($targetFolder, 0755, true);
            }
            $filePath = $targetFolder . DIRECTORY_SEPARATOR . $fileHash;
            
            $handle = fopen($filePath, "w");
            if ($handle) {
                fwrite($handle, $fileData);
                fclose($handle);
                $savedCount++;
            }
        }
        $message = "Success: Data distributed to $savedCount category folders.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rhyzom | Secure Data Distribution</title>
    <style>
        :root {
            --primary: #2563eb;
            --secondary: #1e293b;
            --success: #059669;
            --bg: #f8fafc;
            --text: #334155;
        }
        body { font-family: 'Inter', system-ui, sans-serif; background: var(--bg); color: var(--text); margin: 0; line-height: 1.6; }
        .nav { background: #fff; padding: 1rem 2rem; display: flex; justify-content: space-between; align-items: center; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }
        .nav .logo { font-weight: 800; font-size: 1.5rem; color: var(--primary); text-decoration: none; }
        
        .container { max-width: 900px; margin: 2rem auto; padding: 0 1rem; }
        
        /* Landing Page Styles */
        .hero { text-align: center; padding: 4rem 1rem; }
        .hero h1 { font-size: 3rem; color: var(--secondary); margin-bottom: 1rem; }
        .hero p { font-size: 1.25rem; max-width: 600px; margin: 0 auto 2rem; color: #64748b; }
        .features { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 2rem; margin-top: 3rem; }
        .feature-card { background: #fff; padding: 2rem; border-radius: 12px; box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1); }
        .feature-card h3 { color: var(--primary); margin-top: 0; }

        /* Form Styles */
        .card { background: #fff; padding: 2rem; border-radius: 12px; box-shadow: 0 10px 15px -3px rgba(0,0,0,0.1); }
        .field-group { margin-bottom: 1.5rem; }
        label { display: block; font-weight: 600; margin-bottom: 0.5rem; }
        input[type="text"], textarea { width: 100%; padding: 0.75rem; border: 1px solid #cbd5e1; border-radius: 6px; box-sizing: border-box; font-size: 1rem; }
        textarea { height: 150px; font-family: 'Fira Code', monospace; }
        .cat-inputs { display: grid; grid-template-columns: repeat(auto-fit, minmax(120px, 1fr)); gap: 0.5rem; }
        
        .btn { display: inline-block; padding: 0.75rem 1.5rem; border-radius: 6px; font-weight: 600; text-decoration: none; cursor: pointer; border: none; font-size: 1rem; transition: all 0.2s; }
        .btn-primary { background: var(--primary); color: #fff; }
        .btn-primary:hover { background: #1d4ed8; }
        .btn-outline { border: 2px solid var(--primary); color: var(--primary); }
        .btn-full { width: 100%; }

        .message { padding: 1rem; border-radius: 6px; margin-bottom: 1.5rem; font-weight: 500; }
        .error { background: #fef2f2; color: #991b1b; border: 1px solid #fecaca; }
        .success { background: #ecfdf5; color: #065f46; border: 1px solid #a7f3d0; }
        footer { text-align: center; padding: 2rem; color: #94a3b8; font-size: 0.875rem; }
    </style>
</head>
<body>

<nav class="nav">
    <a href="?view=landing" class="logo">RHYZOM</a>
    <div>
        <a href="?view=landing" style="margin-right: 20px; text-decoration: none; color: var(--text);">Home</a>
        <a href="?view=app" class="btn btn-primary">Open App</a>
    </div>
</nav>

<div class="container">

    <?php if ($currentView === 'landing'): ?>
        <section class="hero">
            <h1>Fragmented Data Storage</h1>
            <p>A secure utility to categorize, hash, and distribute your links and metadata across a structured directory system.</p>
            <a href="?view=app" class="btn btn-primary" style="font-size: 1.2rem;">Get Started Now</a>
        </section>

        <div class="features">
            <div class="feature-card">
                <h3>Categorical Routing</h3>
                <p>Define up to five custom categories. The system automatically creates a synchronized directory structure for organized access.</p>
            </div>
            <div class="feature-card">
                <h3>SHA-256 Hashing</h3>
                <p>Your signature is never stored as plain text. It is converted into a cryptographic hash, serving as a secure, unique filename.</p>
            </div>
            <div class="feature-card">
                <h3>Smart Validation</h3>
                <p>The system distinguishes between your header metadata (like BTC addresses) and resource URLs, ensuring data integrity.</p>
            </div>
        </div>

        <section style="margin-top: 4rem; background: #1e293b; color: #fff; padding: 3rem; border-radius: 12px;">
            <h2>Why use Rhyzom?</h2>
            <p>Rhyzom is designed for users who need a local, lightweight way to archive resource lists across multiple topics simultaneously. By using cryptographic hashes as filenames, you can overwrite and update specific "data nodes" simply by re-entering your signature.</p>
        </section>

    <?php else: ?>
        <div class="card">
            <h2>Data Distribution Tool</h2>
            
            <?php if ($message): ?>
                <div class="message <?php echo strpos($message, 'Error') === 0 ? 'error' : 'success'; ?>">
                    <?php echo $message; ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <div class="field-group">
                    <label>Categories (Max 5 | Max 12 chars | A-Z, 0-9, _)</label>
                    <div class="cat-inputs">
                        <input type="text" name="categories[]" placeholder="Cat_1" maxlength="12" pattern="[A-Za-z0-9_]+">
                        <input type="text" name="categories[]" placeholder="Cat_2" maxlength="12" pattern="[A-Za-z0-9_]+">
                        <input type="text" name="categories[]" placeholder="Cat_3" maxlength="12" pattern="[A-Za-z0-9_]+">
                        <input type="text" name="categories[]" placeholder="Cat_4" maxlength="12" pattern="[A-Za-z0-9_]+">
                        <input type="text" name="categories[]" placeholder="Cat_5" maxlength="12" pattern="[A-Za-z0-9_]+">
                    </div>
                </div>

                <div class="field-group">
                    <label>Content</label>
                    <textarea name="content" placeholder="Line 1: BTC Address or Metadata&#10;Line 2+: https://url-one.com&#10;Line 3+: https://url-two.com"></textarea>
                    <small style="color: #64748b;">Note: First line is metadata; subsequent lines must be valid URLs.</small>
                </div>

                <div class="field-group">
                    <label>Signature (Used to generate SHA-256 Filename)</label>
                    <input type="text" name="signature" maxlength="50" placeholder="Enter your secret signature..." required>
                </div>

                <button type="submit" class="btn btn-primary btn-full">Generate and Distribute Files</button>
            </form>
            
            <p style="text-align: center; margin-top: 1.5rem;">
                <a href="?view=landing" style="color: var(--primary); font-size: 0.9rem;">&larr; Back to Information Page</a>
            </p>
        </div>
    <?php endif; ?>

</div>

<footer>
    &copy; 2026 Rhyzom Data System. All files stored in <code>/rhyzom</code> folder.
</footer>

</body>
</html>