<?php
// Create necessary directories if they don't exist
$dirs = ['tmp_url', 'tmp_url_ip', 'tmp_url_server'];
foreach ($dirs as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// Initialize variables
$message = '';
$messageType = '';

// Function to check if URL is online
function isUrlOnline($url) {
    $headers = @get_headers($url);
    return $headers && strpos($headers[0], '200') !== false;
}

// Function to get IP from URL
function getIpFromUrl($url) {
    $host = parse_url($url, PHP_URL_HOST);
    return gethostbyname($host);
}

// Function to check if line exists in file
function lineExistsInFile($file, $line) {
    if (!file_exists($file)) {
        return false;
    }
    $content = file_get_contents($file);
    $lines = explode("\n", trim($content));
    return in_array(trim($line), array_map('trim', $lines));
}

/**
 * Propagates the URL to all servers listed in servers.txt
 */
function propagateToServers($url, $type) {
    $serversFile = 'servers.txt';
    if (!file_exists($serversFile)) {
        return;
    }

    $servers = file($serversFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($servers as $peer) {
        // Prepare the GET request: peer.com/?type=url
        $separator = (parse_url($peer, PHP_URL_QUERY) == NULL) ? '?' : '&';
        $propagateUrl = $peer . $separator . $type . '=' . urlencode($url);
        
        // Non-blocking-ish: we use a short timeout to prevent hanging the script
        $ctx = stream_context_create(['http' => ['timeout' => 2]]);
        @file_get_contents($propagateUrl, false, $ctx);
    }
}

// Function to process URL submission
function processUrl($url, $type, $userIp) {
    global $message, $messageType;
    
    // Validate URL
    if (empty($url) || !filter_var($url, FILTER_VALIDATE_URL)) {
        $message = 'Invalid URL';
        $messageType = 'error';
        return;
    }
    
    // Validate type
    if ($type !== 'file' && $type !== 'server') {
        $message = 'Invalid type';
        $messageType = 'error';
        return;
    }
    
    // Generate hash for URL - This acts as a deduplication guard 
    // and prevents infinite loops during propagation.
    $urlHash = hash('sha256', $url);
    $hashFile = "tmp_url/{$urlHash}.txt";
    
    // Check if URL hash already exists
    if (file_exists($hashFile)) {
        $message = 'URL already processed';
        $messageType = 'info';
        return;
    }
    
    // Check if URL is online
    if (!isUrlOnline($url)) {
        $message = 'URL is not online';
        $messageType = 'error';
        return;
    }
    
    $targetFile = ($type === 'file') ? 'files.txt' : 'servers.txt';
    
    // Check if URL already exists in target file
    if (lineExistsInFile($targetFile, $url)) {
        $message = 'URL already exists in ' . $targetFile;
        $messageType = 'info';
        return;
    }
    
    // Additional checks for server type
    if ($type === 'server') {
        $userIpHash = hash('sha256', $userIp);
        $userIpHashFile = "tmp_url_ip/{$userIpHash}.txt";
        
        if (file_exists($userIpHashFile)) {
            $message = 'User IP already used';
            $messageType = 'error';
            return;
        }
        
        $urlIp = getIpFromUrl($url);
        $urlIpHash = hash('sha256', $urlIp);
        $urlIpHashFile = "tmp_url_server/{$urlIpHash}.txt";
        
        if (file_exists($urlIpHashFile)) {
            $message = 'Server IP already used';
            $messageType = 'error';
            return;
        }
        
        // Create IP hash files
        touch($userIpHashFile);
        touch($urlIpHashFile);
    }
    
    // Write URL to target file
    $fp = fopen($targetFile, 'a');
    if ($fp) {
        fwrite($fp, $url . "\n");
        fclose($fp);
        
        // Create URL hash file (Mark as processed)
        touch($hashFile);
        
        // NEW: Propagate this URL to all known servers
        propagateToServers($url, $type);
        
        $message = 'URL successfully added and propagated';
        $messageType = 'success';
    } else {
        $message = 'Error writing to file';
        $messageType = 'error';
    }
}

// Get user IP
$userIp = $_SERVER['REMOTE_ADDR'];

// Handle GET parameters
if (isset($_GET['file']) && !empty($_GET['file'])) {
    processUrl($_GET['file'], 'file', $userIp);
} elseif (isset($_GET['server']) && !empty($_GET['server'])) {
    processUrl($_GET['server'], 'server', $userIp);
}

// Handle POST submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $url = isset($_POST['url']) ? trim($_POST['url']) : '';
    $type = isset($_POST['type']) ? $_POST['type'] : '';
    processUrl($url, $type, $userIp);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>URL Submission & Sync</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #f5f5f5;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            padding: 20px;
            padding-top: 80px;
        }

        .nav-bar {
            position: fixed;
            top: 0; left: 0;
            width: 100%; height: 60px;
            background: #fff;
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 0 40px;
            box-shadow: 0 1px 5px rgba(0,0,0,0.05);
            z-index: 1000;
        }

        .nav-logo { font-weight: 600; font-size: 18px; color: #333; text-decoration: none; }
        .nav-links { display: flex; gap: 30px; }
        .nav-links a { text-decoration: none; color: #666; font-size: 14px; text-transform: uppercase; transition: 0.3s; }
        .nav-links a:hover { color: #000; }
        
        .container { width: 100%; max-width: 900px; }
        .form-wrapper { background: white; padding: 50px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.05); }
        h1 { text-align: center; margin-bottom: 40px; color: #333; font-weight: 300; font-size: 32px; }
        
        form { display: flex; flex-direction: column; gap: 20px; }
        input[type="text"] {
            padding: 20px 25px; font-size: 18px; border: 2px solid #e0e0e0; border-radius: 4px;
            outline: none; transition: 0.3s; height: 70px;
        }
        input[type="text"]:focus { border-color: #333; }
        
        .radio-group { display: flex; gap: 20px; justify-content: center; padding: 10px 0; }
        .radio-option { display: flex; align-items: center; gap: 8px; font-size: 18px; cursor: pointer; }
        
        button {
            padding: 20px 40px; font-size: 18px; background: #333; color: white; border: none;
            border-radius: 4px; cursor: pointer; transition: 0.3s; height: 70px;
        }
        button:hover { background: #555; }
        
        .message { padding: 20px; border-radius: 4px; margin-bottom: 20px; text-align: center; }
        .message.success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .message.error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .message.info { background: #d1ecf1; color: #0c5460; border: 1px solid #bee5eb; }

        @media (max-width: 768px) {
            .nav-bar { padding: 0 20px; }
            .nav-links { gap: 15px; }
            .form-wrapper { padding: 30px 20px; }
        }
    </style>
</head>
<body>
    <nav class="nav-bar">
        <a href="#" class="nav-logo">DECENTRALIZED HTTP</a>
        <div class="nav-links">
            <a href="paste.php">Paste</a>
            <a href="files.php">Files</a>
            <a href="search.php">Search</a>
            <a href="index_default.php">Upload</a>
            <a href="readme.html">About</a> 
        </div>
    </nav>

    <div class="container">
        <div class="form-wrapper">
            <h1>URL Submission</h1>
            
            <?php if ($message): ?>
                <div class="message <?php echo $messageType; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            
            <form method="POST">
                <input 
                    type="text" 
                    name="url" 
                    placeholder="https://example.com/cat.jpg" 
                    required
                    value="<?php echo isset($_POST['url']) ? htmlspecialchars($_POST['url']) : ''; ?>"
                >
                
                <div class="radio-group">
                    <label class="radio-option">
                        <input type="radio" name="type" value="file" required <?php echo (!isset($_POST['type']) || $_POST['type'] === 'file') ? 'checked' : ''; ?>>
                        <span>File</span>
                    </label>
                    <label class="radio-option">
                        <input type="radio" name="type" value="server" <?php echo (isset($_POST['type']) && $_POST['type'] === 'server') ? 'checked' : ''; ?>>
                        <span>Server</span>
                    </label>
                </div>
                
                <button type="submit">Submit</button>
            </form>
        </div>
    </div>
</body>
</html>