<?php
// Configuration
define('MAX_FILE_SIZE', 500 * 1024); // 500KB
define('THUMB_MAX_SIZE', 30 * 1024); // 30KB
define('UPLOAD_DIR', 'files/');
define('THUMB_DIR', 'thumb/');
define('IP_DIR', 'ip/');
define('LIKES_DIR', 'likes/');

// Create directories if they don't exist
foreach ([UPLOAD_DIR, THUMB_DIR, IP_DIR, LIKES_DIR] as $dir) {
    if (!file_exists($dir)) {
        mkdir($dir, 0755, true);
    }
}

// Get user IP
$user_ip = $_SERVER['REMOTE_ADDR'];
$ip_hash = hash('sha256', $user_ip);
$ip_file = IP_DIR . $ip_hash;

// Handle file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['image'])) {
    $uploaded_file = $_FILES['image'];
    
    // Check if IP is blocked
    if (!file_exists($ip_file)) {
        
        // Check for upload errors
        if ($uploaded_file['error'] !== UPLOAD_ERR_OK) {
            $upload_message = "Upload error: " . $uploaded_file['error'];
        } 
        // Check file size
        elseif ($uploaded_file['size'] > MAX_FILE_SIZE) {
            $upload_message = "File size exceeds 500KB limit";
        }
        // Check if it's an image
        elseif (!getimagesize($uploaded_file['tmp_name'])) {
            $upload_message = "File is not a valid image";
        }
        else {
            // Generate file hash
            $file_content = file_get_contents($uploaded_file['tmp_name']);
            $file_hash = hash('sha256', $file_content);
            
            // Get original extension
            $original_name = $uploaded_file['name'];
            $extension = pathinfo($original_name, PATHINFO_EXTENSION);
            $filename = $file_hash . ($extension ? '.' . $extension : '');
            $file_path = UPLOAD_DIR . $filename;
            $thumb_path = THUMB_DIR . $filename;
            
            // Check if file already exists
            if (!file_exists($file_path)) {
                // Save original file
                if (move_uploaded_file($uploaded_file['tmp_name'], $file_path)) {
                    $upload_message = "File uploaded successfully!";
                    
                    // Create thumbnail if doesn't exist
                    if (!file_exists($thumb_path)) {
                        createThumbnail($file_path, $thumb_path);
                    }
                    
                    // Create IP block file
                    file_put_contents($ip_file, '');
                } else {
                    $upload_message = "Failed to save file";
                }
            } else {
                $upload_message = "File already exists";
                // Still create IP block file since user attempted upload
                file_put_contents($ip_file, '');
            }
        }
    } else {
        $upload_message = "Upload not allowed for your IP";
    }
}

// Handle like action
if (isset($_GET['like'])) {
    $image_name = basename($_GET['like']);
    $like_dir = LIKES_DIR . $image_name . '/';
    
    if (!file_exists($like_dir)) {
        mkdir($like_dir, 0755, true);
    }
    
    $like_file = $like_dir . $ip_hash;
    if (!file_exists($like_file)) {
        file_put_contents($like_file, '');
    }
    
    // Redirect to avoid form resubmission
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

// Get random thumbnails
function getRandomThumbs($count = 10) {
    $thumbs = glob(THUMB_DIR . '*.{jpg,jpeg,png,gif,webp}', GLOB_BRACE);
    shuffle($thumbs);
    return array_slice($thumbs, 0, $count);
}

// Create compressed thumbnail
function createThumbnail($source_path, $dest_path) {
    $info = getimagesize($source_path);
    if (!$info) return false;
    
    list($width, $height, $type) = $info;
    
    // Create image from source
    switch ($type) {
        case IMAGETYPE_JPEG:
            $source = imagecreatefromjpeg($source_path);
            break;
        case IMAGETYPE_PNG:
            $source = imagecreatefrompng($source_path);
            break;
        case IMAGETYPE_GIF:
            $source = imagecreatefromgif($source_path);
            break;
        case IMAGETYPE_WEBP:
            $source = imagecreatefromwebp($source_path);
            break;
        default:
            return false;
    }
    
    if (!$source) return false;
    
    // Calculate new dimensions (max 800px width while maintaining aspect ratio)
    $max_width = 800;
    $new_width = min($width, $max_width);
    $new_height = ($height / $width) * $new_width;
    
    // Create thumbnail
    $thumbnail = imagecreatetruecolor($new_width, $new_height);
    
    // Preserve transparency for PNG/GIF
    if ($type == IMAGETYPE_PNG || $type == IMAGETYPE_GIF) {
        imagealphablending($thumbnail, false);
        imagesavealpha($thumbnail, true);
        $transparent = imagecolorallocatealpha($thumbnail, 0, 0, 0, 127);
        imagefill($thumbnail, 0, 0, $transparent);
    }
    
    imagecopyresampled($thumbnail, $source, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
    
    // Save with compression
    $quality = 85;
    do {
        // Save to temp file to check size
        $temp_file = tempnam(sys_get_temp_dir(), 'thumb');
        
        switch ($type) {
            case IMAGETYPE_JPEG:
                imagejpeg($thumbnail, $temp_file, $quality);
                break;
            case IMAGETYPE_PNG:
                // PNG compression is 0-9 (0 = no compression, 9 = max)
                $compression = 9 - round(($quality / 100) * 9);
                imagepng($thumbnail, $temp_file, $compression);
                break;
            case IMAGETYPE_GIF:
                imagegif($thumbnail, $temp_file);
                break;
            case IMAGETYPE_WEBP:
                imagewebp($thumbnail, $temp_file, $quality);
                break;
        }
        
        // Reduce quality if file is too large
        if (filesize($temp_file) > THUMB_MAX_SIZE && $quality > 30) {
            $quality -= 10;
            unlink($temp_file);
        } else {
            // Move to destination
            rename($temp_file, $dest_path);
            break;
        }
    } while ($quality > 30);
    
    imagedestroy($source);
    imagedestroy($thumbnail);
    
    return true;
}

// Get random thumbnails for display
$random_thumbs = getRandomThumbs(10);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Image Sharing Platform</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            background-color: #000;
            color: #fff;
            font-family: Arial, sans-serif;
        }
        
        .header {
            background-color: #000;
            padding: 20px;
            text-align: center;
            font-size: 24px;
            font-weight: bold;
            border-bottom: 2px solid #333;
        }
        
        .upload-section {
            background-color: #111;
            padding: 20px;
            margin: 20px auto;
            max-width: 600px;
            border-radius: 8px;
        }
        
        .upload-form input[type="file"] {
            background-color: #222;
            color: #fff;
            padding: 10px;
            border: 1px solid #444;
            border-radius: 4px;
            width: 100%;
        }
        
        .upload-form button {
            background-color: #4CAF50;
            color: white;
            padding: 12px 24px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            margin-top: 10px;
            width: 100%;
            font-size: 16px;
        }
        
        .upload-form button:hover {
            background-color: #45a049;
        }
        
        .message {
            padding: 10px;
            margin: 10px 0;
            border-radius: 4px;
            text-align: center;
        }
        
        .success {
            background-color: #4CAF50;
        }
        
        .error {
            background-color: #f44336;
        }
        
        .gallery {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
            padding: 20px;
        }
        
        .image-container {
            position: relative;
            width: 100%;
            overflow: hidden;
            border-radius: 8px;
            background-color: #111;
        }
        
        .image-container img {
            width: 100%;
            height: auto;
            display: block;
            transition: transform 0.3s ease;
        }
        
        .image-container:hover img {
            transform: scale(1.05);
        }
        
        .like-button {
            position: absolute;
            bottom: 10px;
            right: 10px;
            background-color: #f44336;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
            z-index: 2;
        }
        
        .like-button:hover {
            background-color: #d32f2f;
        }
        
        a {
            text-decoration: none;
        }
    </style>
</head>
<body>
    <div class="header">
        sharing images generated by AI
    </div>
    
    <div class="upload-section">
        <form method="POST" enctype="multipart/form-data" class="upload-form">
            <input type="file" name="image" accept="image/*" required>
            <button type="submit">Upload Image (Max 500KB)</button>
        </form>
        
        <?php if (isset($upload_message)): ?>
            <div class="message <?php echo strpos($upload_message, 'successfully') !== false ? 'success' : 'error'; ?>">
                <?php echo htmlspecialchars($upload_message); ?>
            </div>
        <?php endif; ?>
    </div>
    
    <div class="gallery">
        <?php foreach ($random_thumbs as $thumb_path): 
            $filename = basename($thumb_path);
            $original_path = UPLOAD_DIR . $filename;
        ?>
            <div class="image-container">
                <a href="<?php echo htmlspecialchars($original_path); ?>" target="_blank">
                    <img src="<?php echo htmlspecialchars($thumb_path); ?>" 
                         alt="Image" 
                         onerror="this.style.display='none'">
                </a>
                <form method="GET" style="display: inline;">
                    <input type="hidden" name="like" value="<?php echo htmlspecialchars($filename); ?>">
                    <button type="submit" class="like-button">Like</button>
                </form>
            </div>
        <?php endforeach; ?>
    </div>
</body>
</html>