import javax.swing.*;
import javax.swing.table.DefaultTableModel;
import java.awt.*;
import java.io.*;
import java.net.*;
import java.nio.file.Files;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Scanner;

public class HashCoinV4 extends JFrame {
    
    // --- NESTED BLOCK CLASS (Now inside HashCoinV4) ---
    public static class Block implements Serializable {
        public int index;
        public long timestamp;
        public String data;
        public String previousHash;
        public String hash;

        public Block(int index, String data, String previousHash) {
            this.index = index;
            this.timestamp = System.currentTimeMillis();
            this.data = data;
            this.previousHash = previousHash;
            this.hash = calculateHash();
        }

        public String calculateHash() {
            try {
                String input = index + Long.toString(timestamp) + data + previousHash;
                MessageDigest digest = MessageDigest.getInstance("SHA-256");
                byte[] hashBytes = digest.digest(input.getBytes("UTF-8"));
                StringBuilder hexString = new StringBuilder();
                for (byte b : hashBytes) {
                    String hex = Integer.toHexString(0xff & b);
                    if (hex.length() == 1) hexString.append('0');
                    hexString.append(hex);
                }
                return hexString.toString();
            } catch (Exception e) {
                return "error";
            }
        }
    }

    // --- MAIN APP VARIABLES ---
    private final int PORT = 5000;
    private final String LEDGER_FILE = "ledger_v4.txt";
    private final String CHAIN_FILE = "history_v4.txt";
    private final String SERVERS_FILE = "servers.txt";
    private final String PRIV_KEY_FILE = "private.key";
    private final String PUB_KEY_FILE = "public.key";

    private String userHash;
    private PrivateKey privateKey;
    private PublicKey publicKey;
    private int balance = 100;
    private boolean hasTransacted = false;
    private List<Block> blockchain = new ArrayList<>();
    
    private DefaultTableModel chainModel;
    private JLabel balanceLabel;
    private JTextArea logArea;

    public HashCoinV4() {
        initCryptography();
        loadLocalData();
        loadBlockchain();
        startServer();
        setupUI();
    }

    // --- CRYPTO & LOADING ---
    private void initCryptography() {
        try {
            File privFile = new File(PRIV_KEY_FILE);
            File pubFile = new File(PUB_KEY_FILE);
            if (privFile.exists() && pubFile.exists()) {
                loadKeys();
            } else {
                generateKeys();
            }
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hash = digest.digest(publicKey.getEncoded());
            StringBuilder hexString = new StringBuilder();
            for (byte b : hash) {
                String hex = Integer.toHexString(0xff & b);
                if (hex.length() == 1) hexString.append('0');
                hexString.append(hex);
            }
            this.userHash = hexString.toString();
        } catch (Exception e) { e.printStackTrace(); }
    }

    private void generateKeys() throws Exception {
        KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
        keyGen.initialize(2048);
        KeyPair pair = keyGen.generateKeyPair();
        this.privateKey = pair.getPrivate();
        this.publicKey = pair.getPublic();
        Files.write(new File(PRIV_KEY_FILE).toPath(), privateKey.getEncoded());
        Files.write(new File(PUB_KEY_FILE).toPath(), publicKey.getEncoded());
    }

    private void loadKeys() throws Exception {
        byte[] privBytes = Files.readAllBytes(new File(PRIV_KEY_FILE).toPath());
        byte[] pubBytes = Files.readAllBytes(new File(PUB_KEY_FILE).toPath());
        KeyFactory kf = KeyFactory.getInstance("RSA");
        this.privateKey = kf.generatePrivate(new PKCS8EncodedKeySpec(privBytes));
        this.publicKey = kf.generatePublic(new X509EncodedKeySpec(pubBytes));
    }

    private void loadBlockchain() {
        File f = new File(CHAIN_FILE);
        if (!f.exists() || f.length() == 0) {
            Block genesis = new Block(0, "Genesis Block", "0");
            blockchain.add(genesis);
            saveBlockToFile(genesis);
        } else {
            try (Scanner s = new Scanner(f)) {
                while (s.hasNextLine()) {
                    String line = s.nextLine();
                    String[] p = line.split(",");
                    if (p.length < 5) continue;
                    Block b = new Block(Integer.parseInt(p[0]), p[2], p[3]);
                    b.timestamp = Long.parseLong(p[1]);
                    b.hash = p[4];
                    blockchain.add(b);
                }
            } catch (Exception e) { logToUI("Chain load error: " + e.getMessage()); }
        }
    }

    // --- NETWORKING ---
    private void startServer() {
        new Thread(() -> {
            try (ServerSocket serverSocket = new ServerSocket(PORT)) {
                while (true) {
                    try (Socket clientSocket = serverSocket.accept();
                         BufferedReader in = new BufferedReader(new InputStreamReader(clientSocket.getInputStream()))) {
                        String rawData = in.readLine();
                        if (rawData != null) handleIncomingTransaction(rawData);
                    } catch (Exception e) { }
                }
            } catch (IOException e) { }
        }).start();
    }

    private void handleIncomingTransaction(String rawData) {
        try {
            String[] parts = rawData.split("\\|");
            String from = parts[0], to = parts[1], amt = parts[2], sig = parts[3], pubK = parts[4];

            byte[] pubBytes = Base64.getDecoder().decode(pubK);
            PublicKey senderKey = KeyFactory.getInstance("RSA").generatePublic(new X509EncodedKeySpec(pubBytes));
            Signature rsa = Signature.getInstance("SHA256withRSA");
            rsa.initVerify(senderKey);
            rsa.update((from + "|" + to + "|" + amt).getBytes());

            if (rsa.verify(Base64.getDecoder().decode(sig))) {
                Block lastBlock = blockchain.get(blockchain.size() - 1);
                Block newBlock = new Block(blockchain.size(), from.substring(0,5) + "->" + to.substring(0,5) + ":" + amt, lastBlock.hash);
                blockchain.add(newBlock);
                saveBlockToFile(newBlock);

                if (to.equals(userHash)) {
                    balance += Integer.parseInt(amt);
                    SwingUtilities.invokeLater(() -> balanceLabel.setText(balance + " HC"));
                    saveLedger();
                }

                SwingUtilities.invokeLater(() -> {
                    chainModel.insertRow(0, new Object[]{newBlock.index, newBlock.hash.substring(0,10), "Verified"});
                    logArea.append("Block Added: #" + newBlock.index + "\n");
                });
            }
        } catch (Exception e) { logToUI("Verification Failed: " + e.getMessage()); }
    }

    private void broadcastTransaction(String to, int amt) {
        try {
            String msg = userHash + "|" + to + "|" + amt;
            Signature rsa = Signature.getInstance("SHA256withRSA");
            rsa.initSign(privateKey);
            rsa.update(msg.getBytes());
            String sig = Base64.getEncoder().encodeToString(rsa.sign());
            String pubK = Base64.getEncoder().encodeToString(publicKey.getEncoded());
            String payload = msg + "|" + sig + "|" + pubK;

            File file = new File(SERVERS_FILE);
            if (!file.exists()) return;
            try (Scanner sc = new Scanner(file)) {
                while (sc.hasNextLine()) {
                    String ip = sc.nextLine().trim();
                    if(ip.isEmpty()) continue;
                    new Thread(() -> {
                        try (Socket s = new Socket(ip, PORT); PrintWriter out = new PrintWriter(s.getOutputStream(), true)) {
                            out.println(payload);
                        } catch (Exception e) { }
                    }).start();
                }
            }
        } catch (Exception e) { logToUI("Broadcast Error: " + e.getMessage()); }
    }

    // --- UI & DATA ---
    private void setupUI() {
        setTitle("HashCoinV4 - Blockchain Explorer");
        setSize(900, 600);
        setDefaultCloseOperation(EXIT_ON_CLOSE);
        setLayout(new BorderLayout());

        JPanel nav = new JPanel(new FlowLayout(FlowLayout.LEFT));
        nav.setBackground(new Color(2, 6, 23));
        JLabel logo = new JLabel(" HASHCOIN V4 [BLOCKCHAIN] ");
        logo.setForeground(new Color(100, 255, 218));
        logo.setFont(new Font("Monospaced", Font.BOLD, 22));
        nav.add(logo);
        add(nav, BorderLayout.NORTH);

        JPanel main = new JPanel(new GridLayout(1, 2, 10, 10));
        main.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

        JPanel left = new JPanel();
        left.setLayout(new BoxLayout(left, BoxLayout.Y_AXIS));
        left.add(new JLabel("Your Wallet Address (Public):"));
        JTextField hf = new JTextField(userHash); hf.setEditable(false); left.add(hf);
        
        balanceLabel = new JLabel(balance + " HC");
        balanceLabel.setFont(new Font("SansSerif", Font.BOLD, 50));
        left.add(balanceLabel);

        JTextField toIn = new JTextField(); JTextField amtIn = new JTextField();
        JButton sendBtn = new JButton("Sign & Broadcast");

        sendBtn.addActionListener(e -> {
            try {
                int a = Integer.parseInt(amtIn.getText());
                if(a <= balance) {
                    balance -= a;
                    balanceLabel.setText(balance + " HC");
                    saveLedger();
                    broadcastTransaction(toIn.getText(), a);
                    JOptionPane.showMessageDialog(this, "Transaction Broadcasted!");
                } else {
                    JOptionPane.showMessageDialog(this, "Insufficient balance!");
                }
            } catch (Exception ex) { }
        });

        left.add(new JLabel("Recipient Address:")); left.add(toIn);
        left.add(new JLabel("Amount:")); left.add(amtIn);
        left.add(sendBtn);
        main.add(left);

        JPanel right = new JPanel(new BorderLayout());
        chainModel = new DefaultTableModel(new String[]{"Index", "Hash", "Status"}, 0);
        for(int i = blockchain.size()-1; i>=0; i--) {
            Block b = blockchain.get(i);
            chainModel.addRow(new Object[]{b.index, b.hash.substring(0,10), "Immutable"});
        }
        right.add(new JScrollPane(new JTable(chainModel)), BorderLayout.CENTER);
        logArea = new JTextArea(6, 20); logArea.setBackground(Color.BLACK); logArea.setForeground(Color.CYAN);
        right.add(new JScrollPane(logArea), BorderLayout.SOUTH);
        main.add(right);

        add(main, BorderLayout.CENTER);
    }

    private void saveBlockToFile(Block b) {
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(CHAIN_FILE, true))) {
            bw.write(b.index + "," + b.timestamp + "," + b.data + "," + b.previousHash + "," + b.hash);
            bw.newLine();
        } catch (IOException e) { }
    }

    private void saveLedger() {
        try (PrintWriter out = new PrintWriter(new FileWriter(LEDGER_FILE))) {
            out.println(userHash + ":" + balance + ":" + hasTransacted);
        } catch (IOException e) { }
    }

    private void loadLocalData() {
        File f = new File(LEDGER_FILE);
        if (f.exists()) {
            try (Scanner s = new Scanner(f)) {
                if (s.hasNextLine()) {
                    String[] p = s.nextLine().split(":");
                    this.balance = Integer.parseInt(p[1]);
                    this.hasTransacted = Boolean.parseBoolean(p[2]);
                }
            } catch (Exception e) { }
        }
    }

    private void logToUI(String m) { SwingUtilities.invokeLater(() -> logArea.append(m + "\n")); }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> new HashCoinV4().setVisible(true));
    }
}