import javax.swing.*;
import javax.swing.table.DefaultTableModel;
import java.awt.*;
import java.io.*;
import java.net.*;
import java.nio.file.Files;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.Scanner;

public class HashCoinV3 extends JFrame {
    private final int PORT = 5000;
    private final String LEDGER_FILE = "ledger_v3.txt";
    private final String HISTORY_FILE = "history_v3.txt";
    private final String SERVERS_FILE = "servers.txt";
    private final String PRIV_KEY_FILE = "private.key";
    private final String PUB_KEY_FILE = "public.key";

    private String userHash;
    private PrivateKey privateKey;
    private PublicKey publicKey;
    
    private int balance = 100;
    private boolean hasTransacted = false;
    private DefaultTableModel historyModel;
    private JLabel balanceLabel;
    private JTextArea logArea;

    public HashCoinV3() {
        initCryptography();
        loadLocalData();
        startServer();
        setupUI();
    }

    // --- CRYPTOGRAPHY LOGIC ---
    
    private void initCryptography() {
        try {
            File privFile = new File(PRIV_KEY_FILE);
            File pubFile = new File(PUB_KEY_FILE);

            if (privFile.exists() && pubFile.exists()) {
                loadKeys();
            } else {
                generateKeys();
            }
            
            // Generate Wallet Address (Hash of Public Key)
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hash = digest.digest(publicKey.getEncoded());
            StringBuilder hexString = new StringBuilder();
            for (byte b : hash) {
                String hex = Integer.toHexString(0xff & b);
                if (hex.length() == 1) hexString.append('0');
                hexString.append(hex);
            }
            this.userHash = hexString.toString();
        } catch (Exception e) {
            e.printStackTrace();
            this.userHash = "error_generating_identity";
        }
    }

    private void generateKeys() throws Exception {
        KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
        keyGen.initialize(2048);
        KeyPair pair = keyGen.generateKeyPair();
        this.privateKey = pair.getPrivate();
        this.publicKey = pair.getPublic();

        // Save keys to files
        try (FileOutputStream fosPriv = new FileOutputStream(PRIV_KEY_FILE)) {
            fosPriv.write(privateKey.getEncoded());
        }
        try (FileOutputStream fosPub = new FileOutputStream(PUB_KEY_FILE)) {
            fosPub.write(publicKey.getEncoded());
        }
    }

    private void loadKeys() throws Exception {
        byte[] privBytes = Files.readAllBytes(new File(PRIV_KEY_FILE).toPath());
        byte[] pubBytes = Files.readAllBytes(new File(PUB_KEY_FILE).toPath());

        KeyFactory kf = KeyFactory.getInstance("RSA");
        this.privateKey = kf.generatePrivate(new PKCS8EncodedKeySpec(privBytes));
        this.publicKey = kf.generatePublic(new X509EncodedKeySpec(pubBytes));
    }

    private String signData(String data) throws Exception {
        Signature rsa = Signature.getInstance("SHA256withRSA");
        rsa.initSign(privateKey);
        rsa.update(data.getBytes());
        return Base64.getEncoder().encodeToString(rsa.sign());
    }

    private boolean verifySignature(String data, String signature, PublicKey pubKey) throws Exception {
        Signature rsa = Signature.getInstance("SHA256withRSA");
        rsa.initVerify(pubKey);
        rsa.update(data.getBytes());
        return rsa.verify(Base64.getDecoder().decode(signature));
    }

    // --- NETWORK LOGIC ---

    private void startServer() {
        new Thread(() -> {
            try (ServerSocket serverSocket = new ServerSocket(PORT)) {
                while (true) {
                    try (Socket clientSocket = serverSocket.accept();
                         BufferedReader in = new BufferedReader(new InputStreamReader(clientSocket.getInputStream()))) {
                        
                        String rawData = in.readLine(); // Format: FROM|TO|AMT|SIGNATURE|PUB_KEY
                        if (rawData != null) handleIncomingTransaction(rawData);
                    } catch (Exception e) {
                        System.err.println("Transaction handling error: " + e.getMessage());
                    }
                }
            } catch (IOException e) {
                System.err.println("Server error: " + e.getMessage());
            }
        }).start();
    }

    private void handleIncomingTransaction(String rawData) {
        try {
            String[] parts = rawData.split("\\|");
            String from = parts[0];
            String to = parts[1];
            String amt = parts[2];
            String signature = parts[3];
            String pubKeyStr = parts[4];

            // Reconstruct Public Key from string
            byte[] pubBytes = Base64.getDecoder().decode(pubKeyStr);
            PublicKey senderPubKey = KeyFactory.getInstance("RSA").generatePublic(new X509EncodedKeySpec(pubBytes));

            // 1. Verify Signature
            String message = from + "|" + to + "|" + amt;
            boolean isValid = verifySignature(message, signature, senderPubKey);

            if (!isValid) {
                logToUI("Security Alert: Invalid signature from " + from.substring(0,8));
                return;
            }

            // 2. Process logic
            int amount = Integer.parseInt(amt);
            if (to.equals(userHash)) {
                balance += amount;
                SwingUtilities.invokeLater(() -> balanceLabel.setText(balance + " HC"));
                saveLedger();
            }

            String entry = System.currentTimeMillis() + "," + from + "," + to + "," + amount;
            saveLine(HISTORY_FILE, entry);
            SwingUtilities.invokeLater(() -> {
                historyModel.insertRow(0, new Object[]{"VerifiedRcv", from.substring(0,8), amount});
                logArea.append("Verified: Received from " + from.substring(0,8) + "\n");
            });

        } catch (Exception e) {
            logToUI("Error processing peer transaction.");
        }
    }

    private void broadcastTransaction(String to, int amt) {
        File file = new File(SERVERS_FILE);
        if (!file.exists()) return;

        try {
            String message = userHash + "|" + to + "|" + amt;
            String signature = signData(message);
            String pubKeyEncoded = Base64.getEncoder().encodeToString(publicKey.getEncoded());
            String fullPayload = message + "|" + signature + "|" + pubKeyEncoded;

            try (Scanner scanner = new Scanner(file)) {
                while (scanner.hasNextLine()) {
                    String peerIP = scanner.nextLine().trim();
                    if (peerIP.isEmpty()) continue;

                    new Thread(() -> {
                        try (Socket socket = new Socket(peerIP, PORT);
                             PrintWriter out = new PrintWriter(socket.getOutputStream(), true)) {
                            out.println(fullPayload);
                        } catch (IOException e) {
                            logToUI("Node offline: " + peerIP);
                        }
                    }).start();
                }
            }
        } catch (Exception e) {
            logToUI("Signing Error: " + e.getMessage());
        }
    }

    // --- UI & UTILS ---

    private void setupUI() {
        setTitle("HashCoinV3 - Secure P2P");
        setSize(850, 600);
        setDefaultCloseOperation(EXIT_ON_CLOSE);
        setLayout(new BorderLayout());

        JPanel nav = new JPanel(new FlowLayout(FlowLayout.LEFT));
        nav.setBackground(new Color(30, 41, 59));
        JLabel logo = new JLabel(" HASHCOIN V3 [SECURE] ");
        logo.setForeground(Color.CYAN);
        logo.setFont(new Font("Monospaced", Font.BOLD, 20));
        nav.add(logo);
        add(nav, BorderLayout.NORTH);

        JPanel main = new JPanel(new GridLayout(1, 2, 10, 10));
        main.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

        JPanel left = new JPanel();
        left.setLayout(new BoxLayout(left, BoxLayout.Y_AXIS));
        left.add(new JLabel("Your RSA Wallet Address (Public Hash):"));
        JTextField hashField = new JTextField(userHash);
        hashField.setEditable(false);
        left.add(hashField);

        balanceLabel = new JLabel(balance + " HC");
        balanceLabel.setFont(new Font("SansSerif", Font.BOLD, 45));
        left.add(balanceLabel);

        JTextField toInput = new JTextField();
        JTextField amtInput = new JTextField();
        JButton sendBtn = new JButton("Sign & Broadcast Transaction");
        if(hasTransacted) sendBtn.setEnabled(false);

        sendBtn.addActionListener(e -> {
            try {
                int amt = Integer.parseInt(amtInput.getText());
                if(amt <= balance) {
                    balance -= amt;
                    hasTransacted = true;
                    balanceLabel.setText(balance + " HC");
                    sendBtn.setEnabled(false);
                    saveLedger();
                    broadcastTransaction(toInput.getText(), amt);
                    JOptionPane.showMessageDialog(this, "Signed with Private Key and Broadcasted!");
                }
            } catch (Exception ex) {
                JOptionPane.showMessageDialog(this, "Invalid input.");
            }
        });

        left.add(new JLabel("Recipient Wallet Address:"));
        left.add(toInput);
        left.add(new JLabel("Amount to Send:"));
        left.add(amtInput);
        left.add(sendBtn);
        main.add(left);

        JPanel right = new JPanel(new BorderLayout());
        logArea = new JTextArea(8, 20);
        logArea.setEditable(false);
        logArea.setBackground(Color.BLACK);
        logArea.setForeground(new Color(50, 255, 50));
        
        historyModel = new DefaultTableModel(new String[]{"Security", "Sender", "HC"}, 0);
        right.add(new JScrollPane(new JTable(historyModel)), BorderLayout.CENTER);
        right.add(new JScrollPane(logArea), BorderLayout.SOUTH);
        main.add(right);

        add(main, BorderLayout.CENTER);
        add(new JLabel(" RSA keys active. Transactions are cryptographically signed.", SwingConstants.CENTER), BorderLayout.SOUTH);
    }

    private void logToUI(String msg) {
        SwingUtilities.invokeLater(() -> logArea.append(msg + "\n"));
    }

    private void saveLedger() {
        try (PrintWriter out = new PrintWriter(new FileWriter(LEDGER_FILE))) {
            out.println(userHash + ":" + balance + ":" + hasTransacted);
        } catch (IOException e) { e.printStackTrace(); }
    }

    private void saveLine(String file, String line) {
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(file, true))) {
            bw.write(line); bw.newLine();
        } catch (IOException e) { e.printStackTrace(); }
    }

    private void loadLocalData() {
        File f = new File(LEDGER_FILE);
        if (f.exists()) {
            try (Scanner s = new Scanner(f)) {
                if (s.hasNextLine()) {
                    String[] parts = s.nextLine().split(":");
                    this.balance = Integer.parseInt(parts[1]);
                    this.hasTransacted = Boolean.parseBoolean(parts[2]);
                }
            } catch (Exception e) { e.printStackTrace(); }
        }
    }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> new HashCoinV3().setVisible(true));
    }
}