import javax.swing.*;
import javax.swing.table.DefaultTableModel;
import java.awt.*;
import java.io.*;
import java.net.*;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;

public class HashCoinV2 extends JFrame {
    private final int PORT = 5000; // Communication port
    private final String LEDGER_FILE = "ledger_v2.txt";
    private final String HISTORY_FILE = "history_v2.txt";
    private final String SERVERS_FILE = "servers.txt";

    private String userHash;
    private int balance = 100;
    private boolean hasTransacted = false;
    private DefaultTableModel historyModel;
    private JLabel balanceLabel;
    private JTextArea logArea;

    public HashCoinV2() {
        initIdentity();
        loadLocalData();
        startServer(); // Start listening for peers
        setupUI();
    }

    private void initIdentity() {
        try {
            String ip = InetAddress.getLocalHost().getHostAddress();
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hash = digest.digest(ip.getBytes("UTF-8"));
            StringBuilder hexString = new StringBuilder();
            for (byte b : hash) {
                String hex = Integer.toHexString(0xff & b);
                if (hex.length() == 1) hexString.append('0');
                hexString.append(hex);
            }
            this.userHash = hexString.toString();
        } catch (Exception e) { this.userHash = "error_hash"; }
    }

    // --- NETWORK LOGIC: THE SERVER ---
    private void startServer() {
        new Thread(() -> {
            try (ServerSocket serverSocket = new ServerSocket(PORT)) {
                while (true) {
                    try (Socket clientSocket = serverSocket.accept();
                         BufferedReader in = new BufferedReader(new InputStreamReader(clientSocket.getInputStream()))) {
                        
                        String data = in.readLine(); // Format: FROM_HASH|TO_HASH|AMOUNT
                        if (data != null) handleIncomingTransaction(data);
                    }
                }
            } catch (IOException e) {
                System.err.println("Server error: " + e.getMessage());
            }
        }).start();
    }

    private void handleIncomingTransaction(String data) {
        String[] parts = data.split("\\|");
        String from = parts[0];
        String to = parts[1];
        int amt = Integer.parseInt(parts[2]);

        // If I am the receiver, update my balance
        if (to.equals(userHash)) {
            balance += amt;
            SwingUtilities.invokeLater(() -> balanceLabel.setText(balance + " HC"));
            saveLedger();
        }

        // Log to history and UI
        String entry = System.currentTimeMillis() + "," + from + "," + to + "," + amt;
        saveLine(HISTORY_FILE, entry);
        SwingUtilities.invokeLater(() -> {
            historyModel.insertRow(0, new Object[]{"NetRcv", from.substring(0,8), amt});
            logArea.append("Sync: Received TX from " + from.substring(0,8) + "\n");
        });
    }

    // --- NETWORK LOGIC: THE CLIENT (BROADCAST) ---
    private void broadcastTransaction(String to, int amt) {
        File file = new File(SERVERS_FILE);
        if (!file.exists()) return;

        try (Scanner scanner = new Scanner(file)) {
            while (scanner.hasNextLine()) {
                String peerIP = scanner.nextLine().trim();
                if (peerIP.isEmpty()) continue;

                new Thread(() -> {
                    try (Socket socket = new Socket(peerIP, PORT);
                         PrintWriter out = new PrintWriter(socket.getOutputStream(), true)) {
                        out.println(userHash + "|" + to + "|" + amt);
                    } catch (IOException e) {
                        SwingUtilities.invokeLater(() -> logArea.append("Node offline: " + peerIP + "\n"));
                    }
                }).start();
            }
        } catch (FileNotFoundException e) { e.printStackTrace(); }
    }

    // --- UI & DATA PERSISTENCE ---
    private void setupUI() {
        setTitle("HashCoinV2 - P2P Network");
        setSize(800, 600);
        setDefaultCloseOperation(EXIT_ON_CLOSE);
        setLayout(new BorderLayout());

        // Navigation
        JPanel nav = new JPanel(new FlowLayout(FlowLayout.LEFT));
        nav.setBackground(new Color(15, 23, 42));
        JLabel logo = new JLabel(" HASHCOIN V2 ");
        logo.setForeground(Color.CYAN);
        logo.setFont(new Font("Monospaced", Font.BOLD, 20));
        nav.add(logo);
        add(nav, BorderLayout.NORTH);

        JPanel main = new JPanel(new GridLayout(1, 2, 10, 10));
        main.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

        // Left Panel: Wallet
        JPanel left = new JPanel();
        left.setLayout(new BoxLayout(left, BoxLayout.Y_AXIS));
        left.add(new JLabel("Your Hash Address:"));
        JTextField hashField = new JTextField(userHash);
        hashField.setEditable(false);
        left.add(hashField);

        balanceLabel = new JLabel(balance + " HC");
        balanceLabel.setFont(new Font("SansSerif", Font.BOLD, 40));
        left.add(balanceLabel);

        JTextField toInput = new JTextField();
        JTextField amtInput = new JTextField();
        JButton sendBtn = new JButton("Broadcast Transaction");
        if(hasTransacted) sendBtn.setEnabled(false);

        sendBtn.addActionListener(e -> {
            int amt = Integer.parseInt(amtInput.getText());
            if(amt <= balance) {
                balance -= amt;
                hasTransacted = true;
                balanceLabel.setText(balance + " HC");
                sendBtn.setEnabled(false);
                saveLedger();
                broadcastTransaction(toInput.getText(), amt);
                JOptionPane.showMessageDialog(this, "Broadcasted to peers!");
            }
        });

        left.add(new JLabel("Recipient Hash:"));
        left.add(toInput);
        left.add(new JLabel("Amount:"));
        left.add(amtInput);
        left.add(sendBtn);
        main.add(left);

        // Right Panel: Logs & History
        JPanel right = new JPanel(new BorderLayout());
        logArea = new JTextArea(5, 20);
        logArea.setEditable(false);
        logArea.setBackground(Color.BLACK);
        logArea.setForeground(Color.GREEN);
        
        historyModel = new DefaultTableModel(new String[]{"Type", "Hash", "HC"}, 0);
        right.add(new JScrollPane(new JTable(historyModel)), BorderLayout.CENTER);
        right.add(new JScrollPane(logArea), BorderLayout.SOUTH);
        main.add(right);

        add(main, BorderLayout.CENTER);
        add(new JLabel(" Consensus nodes loaded from servers.txt", SwingConstants.CENTER), BorderLayout.SOUTH);
    }

    private void saveLedger() {
        try (PrintWriter out = new PrintWriter(new FileWriter(LEDGER_FILE))) {
            out.println(userHash + ":" + balance + ":" + hasTransacted);
        } catch (IOException e) { e.printStackTrace(); }
    }

    private void saveLine(String file, String line) {
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(file, true))) {
            bw.write(line); bw.newLine();
        } catch (IOException e) { e.printStackTrace(); }
    }

    private void loadLocalData() {
        File f = new File(LEDGER_FILE);
        if (f.exists()) {
            try (Scanner s = new Scanner(f)) {
                if (s.hasNextLine()) {
                    String[] parts = s.nextLine().split(":");
                    this.balance = Integer.parseInt(parts[1]);
                    this.hasTransacted = Boolean.parseBoolean(parts[2]);
                }
            } catch (Exception e) { e.printStackTrace(); }
        }
    }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> new HashCoinV2().setVisible(true));
    }
}