import javax.swing.*;
import javax.swing.table.DefaultTableModel;
import java.awt.*;
import java.io.*;
import java.net.InetAddress;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;

public class HashCoinApp extends JFrame {
    // Data Storage Files
    private final String LEDGER_FILE = "ledger.txt";
    private final String HISTORY_FILE = "history.txt";
    private final String SERVERS_FILE = "servers.txt";

    private String userHash;
    private int balance = 100;
    private boolean hasTransacted = false;
    private DefaultTableModel historyModel;
    private JLabel balanceLabel;

    public HashCoinApp() {
        initIdentity();
        loadLocalData();
        checkConsensus(); // Consensus logic
        setupUI();
    }

    // 1. IDENTITY: SHA-256 of IP
    private void initIdentity() {
        try {
            String ip = InetAddress.getLocalHost().getHostAddress();
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hash = digest.digest(ip.getBytes("UTF-8"));
            StringBuilder hexString = new StringBuilder();
            for (byte b : hash) {
                String hex = Integer.toHexString(0xff & b);
                if (hex.length() == 1) hexString.append('0');
                hexString.append(hex);
            }
            this.userHash = hexString.toString();
        } catch (Exception e) {
            this.userHash = "default_hash_error";
        }
    }

    // 2. CONSENSUS: Read servers.txt
    private void checkConsensus() {
        File file = new File(SERVERS_FILE);
        List<String> nodes = new ArrayList<>();
        if (file.exists()) {
            try (Scanner scanner = new Scanner(file)) {
                while (scanner.hasNextLine()) nodes.add(scanner.nextLine());
            } catch (FileNotFoundException e) { e.printStackTrace(); }
        }
        
        // Simulation: If this hash is found in a "Global Registry" (simulated by servers list)
        // In a real app, you'd perform a Socket request to these IPs to verify the transaction.
        System.out.println("Consensus reached with " + nodes.size() + " nodes.");
    }

    // 3. UI SETUP
    private void setupUI() {
        setTitle("HashCoin Testnet - Java Edition");
        setSize(700, 600);
        setDefaultCloseOperation(EXIT_ON_CLOSE);
        setLayout(new BorderLayout());

        // TOP MENU
        JPanel header = new JPanel(new FlowLayout(FlowLayout.LEFT));
        header.setBackground(new Color(30, 41, 59));
        JLabel title = new JLabel(" HASHCOIN ");
        title.setForeground(Color.WHITE);
        title.setFont(new Font("SansSerif", Font.BOLD, 18));
        header.add(title);
        header.add(createNavButton("Dashboard"));
        header.add(createNavButton("History"));
        add(header, BorderLayout.NORTH);

        // MAIN CONTENT
        JPanel content = new JPanel();
        content.setLayout(new BoxLayout(content, BoxLayout.Y_AXIS));
        content.setBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20));

        // Wallet Card
        JPanel walletCard = new JPanel(new GridLayout(0, 1));
        walletCard.setBorder(BorderFactory.createTitledBorder("Your Wallet"));
        walletCard.add(new JLabel("Address: " + userHash.substring(0, 32) + "..."));
        balanceLabel = new JLabel(balance + " HC");
        balanceLabel.setFont(new Font("SansSerif", Font.BOLD, 30));
        balanceLabel.setForeground(new Color(37, 99, 235));
        walletCard.add(balanceLabel);
        content.add(walletCard);

        // Transaction Form
        JPanel formCard = new JPanel(new FlowLayout(FlowLayout.LEFT));
        formCard.setBorder(BorderFactory.createTitledBorder("Send Transaction (1 Remaining)"));
        JTextField toInput = new JTextField(20);
        JTextField amtInput = new JTextField(5);
        JButton sendBtn = new JButton("Transfer");
        
        if(hasTransacted) sendBtn.setEnabled(false);

        sendBtn.addActionListener(e -> handleTransaction(toInput.getText(), amtInput.getText(), sendBtn));
        
        formCard.add(new JLabel("To:"));
        formCard.add(toInput);
        formCard.add(new JLabel("Amt:"));
        formCard.add(amtInput);
        formCard.add(sendBtn);
        content.add(formCard);

        // History Table
        String[] cols = {"Time", "To", "Amount"};
        historyModel = new DefaultTableModel(cols, 0);
        JTable table = new JTable(historyModel);
        content.add(new JLabel("Recent Transactions:"));
        content.add(new JScrollPane(table));

        add(content, BorderLayout.CENTER);

        // FOOTER
        JPanel footer = new JPanel();
        footer.add(new JLabel("Java Core v1.0 | Connected to servers.txt nodes"));
        add(footer, BorderLayout.SOUTH);

        loadHistoryUI();
    }

    private JButton createNavButton(String text) {
        JButton b = new JButton(text);
        b.setContentAreaFilled(false);
        b.setForeground(Color.LIGHT_GRAY);
        b.setBorderPainted(false);
        return b;
    }

    // 4. TRANSACTION LOGIC
    private void handleTransaction(String to, String amtStr, JButton btn) {
        try {
            int amt = Integer.parseInt(amtStr);
            if (amt > balance) throw new Exception("Low balance");
            
            balance -= amt;
            hasTransacted = true;
            balanceLabel.setText(balance + " HC");
            btn.setEnabled(false);

            // Save to files
            String entry = System.currentTimeMillis() + "," + to + "," + amt;
            saveLine(HISTORY_FILE, entry);
            saveLedger();
            
            historyModel.insertRow(0, new Object[]{"Just now", to, amt});
            JOptionPane.showMessageDialog(this, "Transaction Broadcasted!");
        } catch (Exception ex) {
            JOptionPane.showMessageDialog(this, "Error: " + ex.getMessage());
        }
    }

    private void saveLine(String file, String line) {
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(file, true))) {
            bw.write(line);
            bw.newLine();
        } catch (IOException e) { e.printStackTrace(); }
    }

    private void saveLedger() {
        try (PrintWriter out = new PrintWriter(new FileWriter(LEDGER_FILE))) {
            out.println(userHash + ":" + balance + ":" + hasTransacted);
        } catch (IOException e) { e.printStackTrace(); }
    }

    private void loadLocalData() {
        File f = new File(LEDGER_FILE);
        if (f.exists()) {
            try (Scanner s = new Scanner(f)) {
                if (s.hasNextLine()) {
                    String[] parts = s.nextLine().split(":");
                    this.balance = Integer.parseInt(parts[1]);
                    this.hasTransacted = Boolean.parseBoolean(parts[2]);
                }
            } catch (Exception e) { e.printStackTrace(); }
        }
    }

    private void loadHistoryUI() {
        File f = new File(HISTORY_FILE);
        if (f.exists()) {
            try (Scanner s = new Scanner(f)) {
                while (s.hasNextLine()) {
                    String[] p = s.nextLine().split(",");
                    historyModel.insertRow(0, new Object[]{p[0], p[1], p[2]});
                }
            } catch (Exception e) { e.printStackTrace(); }
        }
    }

    public static void main(String[] args) {
        SwingUtilities.invokeLater(() -> new HashCoinApp().setVisible(true));
    }
}