import java.io.*;
import java.net.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.*;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class VerifiedServerFileMerger {

    // Configuration Constants
    private static final String SERVERS_LIST_FILE = "servers.txt";
    private static final String MERGE_DIR = "merge_files";
    private static final String MASTER_OUTPUT_FILE = "files.txt";
    private static final String SERVER_TARGET_FILE = "files.txt"; 

    public static void main(String[] args) {
        try {
            // 1. Setup Directories
            Path mergeDirPath = Paths.get(MERGE_DIR);
            if (!Files.exists(mergeDirPath)) {
                Files.createDirectories(mergeDirPath);
            }

            // 2. Read Servers List
            // FIX: The method signature for readServersList() was missing 'static' 
            List<String> servers = readServersList(); 
            if (servers.isEmpty()) {
                System.out.println("No servers found in " + SERVERS_LIST_FILE);
                return;
            }

            // 3. Process each server (Download, Verify Hashes, Save)
            System.out.println("--- Starting Downloads & Verification ---");
            for (String serverUrl : servers) {
                processServer(serverUrl, mergeDirPath);
            }

            // 4. Merge Files
            System.out.println("\n--- Starting Final Merge Process ---");
            mergeDownloadedFiles(mergeDirPath);

            System.out.println("\nProcess Completed Successfully.");

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    // This method has been updated to include the 'static' modifier.
    private static List<String> readServersList() {
        try {
            Path path = Paths.get(SERVERS_LIST_FILE);
            if (!Files.exists(path)) return Collections.emptyList();
            return Files.readAllLines(path).stream()
                    .map(String::trim)
                    .filter(line -> !line.isEmpty())
                    .collect(Collectors.toList());
        } catch (IOException e) {
            System.err.println("Error reading servers list: " + e.getMessage());
            return Collections.emptyList();
        }
    }
    
    // ... (rest of the class methods: processServer, verifyUrlContentHash, 
    //      mergeDownloadedFiles, calculateStringSha256, calculateStreamSha256, 
    //      bytesToHex) ...
    
    // --- Remaining methods from the previous response are defined below ---

    private static void processServer(String baseUrl, Path outputDir) {
        try {
            // Construct URL to files.txt
            String fullUrlStr = baseUrl.endsWith("/") ? baseUrl + SERVER_TARGET_FILE : baseUrl + "/" + SERVER_TARGET_FILE;
            URL url = new URL(fullUrlStr);

            // Get IP Address & Generate Filename based on IP Hash
            InetAddress address = InetAddress.getByName(url.getHost());
            String ipAddress = address.getHostAddress();
            String ipHash = calculateStringSha256(ipAddress);
            Path outputPath = outputDir.resolve(ipHash + ".txt");

            System.out.println("Server: " + baseUrl + " (" + ipAddress + ")");

            // 1. Download the list file first
            List<String> rawLines;
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(url.openStream()))) {
                rawLines = reader.lines().collect(Collectors.toList());
                System.out.println("  [Info] Downloaded list. Checking " + rawLines.size() + " lines...");
            }

            // 2. Verify each line in the downloaded list
            List<String> verifiedLines = new ArrayList<>();
            for (String lineUrl : rawLines) {
                if (verifyUrlContentHash(lineUrl)) {
                    verifiedLines.add(lineUrl);
                }
            }

            // 3. Save only the verified lines to the merge folder
            if (!verifiedLines.isEmpty()) {
                Files.write(outputPath, verifiedLines, StandardCharsets.UTF_8);
                System.out.println("  [Success] Saved " + verifiedLines.size() + " valid lines to " + outputPath.getFileName());
            } else {
                System.out.println("  [Warn] All lines failed verification. Nothing saved.");
            }

        } catch (UnknownHostException e) {
            System.err.println("  [ERR] Could not resolve IP for: " + baseUrl);
        } catch (IOException e) {
            System.err.println("  [ERR] Connection failed: " + baseUrl + " (" + e.getMessage() + ")");
        }
    }

    private static boolean verifyUrlContentHash(String urlString) {
        try {
            String trimmedUrl = urlString.trim();
            if (trimmedUrl.isEmpty()) return false;

            URL fileUrl = new URL(trimmedUrl);
            
            // Extract filename (e.g., "123.jpg")
            String path = fileUrl.getPath();
            String fileName = path.substring(path.lastIndexOf('/') + 1);
            
            // Extract expected hash (remove extension, e.g., "123")
            String expectedHash = fileName;
            if (fileName.contains(".")) {
                expectedHash = fileName.substring(0, fileName.lastIndexOf('.'));
            }

            // Download stream and calculate hash on the fly
            try (InputStream in = fileUrl.openStream()) {
                String actualHash = calculateStreamSha256(in);
                
                if (actualHash.equalsIgnoreCase(expectedHash)) {
                    return true;
                } else {
                    System.out.println("    [Skip] Hash mismatch: " + fileName); 
                    return false;
                }
            }

        } catch (Exception e) {
            System.out.println("    [Skip] Error checking URL: " + urlString + " (" + e.getMessage() + ")");
            return false;
        }
    }

    private static void mergeDownloadedFiles(Path mergeDirPath) throws IOException {
        Path masterFile = Paths.get(MASTER_OUTPUT_FILE);
        
        Set<String> existingLines = new HashSet<>();
        if (Files.exists(masterFile)) {
            existingLines.addAll(Files.readAllLines(masterFile));
        }

        List<String> linesToAppend = new ArrayList<>();

        try (Stream<Path> stream = Files.list(mergeDirPath)) {
            stream.filter(file -> !Files.isDirectory(file))
                  .forEach(path -> {
                      try {
                          List<String> fileLines = Files.readAllLines(path);
                          for (String line : fileLines) {
                              String trimmed = line.trim();
                              if (!trimmed.isEmpty() && !existingLines.contains(trimmed)) {
                                  linesToAppend.add(trimmed);
                                  existingLines.add(trimmed); 
                              }
                          }
                      } catch (IOException e) {
                          System.err.println("Error reading file: " + path);
                      }
                  });
        }

        if (!linesToAppend.isEmpty()) {
            StandardOpenOption option = Files.exists(masterFile) ? StandardOpenOption.APPEND : StandardOpenOption.CREATE;
            Files.write(masterFile, linesToAppend, StandardCharsets.UTF_8, option);
            System.out.println("Added " + linesToAppend.size() + " new unique lines to " + MASTER_OUTPUT_FILE);
        } else {
            System.out.println("No new unique lines found to merge.");
        }
    }

    private static String calculateStringSha256(String input) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            return bytesToHex(digest.digest(input.getBytes(StandardCharsets.UTF_8)));
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    private static String calculateStreamSha256(InputStream inputStream) throws IOException, NoSuchAlgorithmException {
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        byte[] buffer = new byte[8192]; 
        int bytesRead;
        while ((bytesRead = inputStream.read(buffer)) != -1) {
            digest.update(buffer, 0, bytesRead);
        }
        return bytesToHex(digest.digest());
    }

    private static String bytesToHex(byte[] hash) {
        StringBuilder hexString = new StringBuilder(2 * hash.length);
        for (byte b : hash) {
            String hex = Integer.toHexString(0xff & b);
            if (hex.length() == 1) {
                hexString.append('0');
            }
            hexString.append(hex);
        }
        return hexString.toString();
    }
}