﻿import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.stream.Collectors;

public class UnifiedRepeatCounter {

    private static final String FILENAME_TO_READ = "files.txt";
    private static final int MAX_RESULTS = 100;

    public static void main(String[] args) {
        // Maps to store counts
        Map<String, Integer> serverCounts = new HashMap<>();
        Map<String, Integer> fileCounts = new HashMap<>();

        System.out.println("Starting analysis of " + FILENAME_TO_READ + "...");

        // 1. Read the file and populate the maps
        try (BufferedReader br = new BufferedReader(new FileReader(FILENAME_TO_READ))) {
            String line;
            while ((line = br.readLine()) != null) {
                // Extract and count Server (Protocol + Host)
                String baseUrl = extractServerBaseUrl(line);
                if (!baseUrl.isEmpty()) {
                    serverCounts.put(baseUrl, serverCounts.getOrDefault(baseUrl, 0) + 1);
                }

                // Extract and count File (Filename only)
                String filename = extractFilename(line);
                if (!filename.isEmpty()) {
                    fileCounts.put(filename, fileCounts.getOrDefault(filename, 0) + 1);
                }
            }
        } catch (IOException e) {
            System.err.println("❌ Error reading the file: " + e.getMessage());
            waitForEnter(); // Wait for user even on error before exiting
            return;
        }

        // --- Analysis and Output ---
        
        // 2. Analyze and print the most repeated Servers
        System.out.println("\n" + "-".repeat(60));
        analyzeAndPrintResults(serverCounts, "Servers");
        System.out.println("-".repeat(60));

        // 3. Analyze and print the most repeated Files
        System.out.println("\n" + "-".repeat(60));
        analyzeAndPrintResults(fileCounts, "Files");
        System.out.println("-".repeat(60));

        // 4. Wait for user input before program exit
        waitForEnter();
    }

    // --- Helper Methods ---

    /**
     * Sorts, limits, and prints the results for either servers or files.
     */
    private static void analyzeAndPrintResults(Map<String, Integer> counts, String type) {
        // Sort the map by count in descending order (Most repeated first)
        Map<String, Integer> sortedByCount = counts.entrySet()
                .stream()
                .sorted(Map.Entry.comparingByValue(Comparator.reverseOrder())) 
                .limit(MAX_RESULTS) // Limit to the top 100 results
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        Map.Entry::getValue,
                        (oldValue, newValue) -> oldValue, 
                        LinkedHashMap::new 
                ));

        System.out.println("? Top " + MAX_RESULTS + " Most Repeated " + type + " (Descending Order):");
        sortedByCount.forEach((item, count) -> {
            System.out.println(item + ", " + count);
        });
    }
    
    /**
     * Extracts the base server URL (protocol + host) from a full URL string.
     */
    private static String extractServerBaseUrl(String urlString) {
        if (urlString == null || urlString.trim().isEmpty()) {
            return "";
        }
        try {
            URL url = new URL(urlString.trim());
            String protocol = url.getProtocol();
            String host = url.getHost();

            if (protocol != null && !protocol.isEmpty() && host != null && !host.isEmpty()) {
                return protocol + "://" + host;
            }

        } catch (MalformedURLException e) {
            // Silently ignore malformed URLs that can't be parsed by URL class
        }
        return ""; 
    }

    /**
     * Extracts the filename from a URL or file path by finding the part after the last '/'.
     */
    private static String extractFilename(String url) {
        if (url == null || url.isEmpty()) {
            return "";
        }
        int lastSlashIndex = url.lastIndexOf('/');
        
        if (lastSlashIndex != -1 && lastSlashIndex < url.length() - 1) {
            return url.substring(lastSlashIndex + 1).trim();
        }
        return url.trim();
    }

    /**
     * Pauses program execution until the user presses the Enter key.
     */
    private static void waitForEnter() {
        System.out.println("\nPress ENTER to exit the program...");
        try {
            // Read a single line from the standard input
            new BufferedReader(new InputStreamReader(System.in)).readLine();
        } catch (IOException e) {
            // This error is unlikely but handled gracefully
            System.err.println("Error reading input: " + e.getMessage());
        }
    }
}