import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.stream.Collectors;

public class ServerRepeatCounter {

    private static final String FILENAME_TO_READ = "files.txt";
    private static final int MAX_RESULTS = 100;

    public static void main(String[] args) {
        // 1. Map to store server base URLs and their counts
        Map<String, Integer> serverCounts = new HashMap<>();

        // 2. Read the file and populate the map
        System.out.println("Reading file: " + FILENAME_TO_READ + " and counting server occurrences...");
        try (BufferedReader br = new BufferedReader(new FileReader(FILENAME_TO_READ))) {
            String line;
            while ((line = br.readLine()) != null) {
                String baseUrl = extractServerBaseUrl(line);
                if (!baseUrl.isEmpty()) {
                    serverCounts.put(baseUrl, serverCounts.getOrDefault(baseUrl, 0) + 1);
                }
            }
        } catch (IOException e) {
            System.err.println("An error occurred while reading the file: " + e.getMessage());
            // Exit if the file cannot be read
            return;
        }

        // 3. Sort the map by count in descending order (Most repeated first)
        Map<String, Integer> sortedByCount = serverCounts.entrySet()
                .stream()
                // Sort by the map value (count) in reverse/descending order
                .sorted(Map.Entry.comparingByValue(Comparator.reverseOrder())) 
                .limit(MAX_RESULTS) // Limit to the top 100 results (most repeated)
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        Map.Entry::getValue,
                        (oldValue, newValue) -> oldValue, 
                        LinkedHashMap::new // Use LinkedHashMap to maintain the sorting order
                ));

        // 4. Print the results
        System.out.println("\n--- Top " + MAX_RESULTS + " Most Repeated Servers (in descending order of count) ---");
        sortedByCount.forEach((serverUrl, count) -> {
            System.out.println(serverUrl + ", " + count);
        });
        System.out.println("---------------------------------------------------------------------");
    }

    /**
     * Extracts the base server URL (protocol + host) from a full URL string.
     * Example: "https://3gp.neocities.org/files/1234.jpg" -> "https://3gp.neocities.org"
     * @param urlString The line read from the file.
     * @return The extracted base URL, or an empty string if parsing fails.
     */
    private static String extractServerBaseUrl(String urlString) {
        if (urlString == null || urlString.trim().isEmpty()) {
            return "";
        }
        try {
            // Use the standard Java URL class for reliable parsing
            URL url = new URL(urlString.trim());
            
            // Reconstruct the base URL: Protocol + "://" + Host
            String protocol = url.getProtocol();
            String host = url.getHost();

            if (protocol != null && !protocol.isEmpty() && host != null && !host.isEmpty()) {
                return protocol + "://" + host;
            }

        } catch (MalformedURLException e) {
            // Handle lines that are not valid URLs (e.g., local file paths, malformed text)
            // System.err.println("Skipping malformed URL: " + urlString); // Uncomment for debugging
        }
        return ""; // Return empty string if it's not a valid URL or parsing failed
    }
}