import java.io.*;
import java.net.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.*;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class ServerFileMerger {

    // Configuration Constants
    private static final String SERVERS_LIST_FILE = "servers.txt";
    private static final String MERGE_DIR = "merge_files";
    private static final String MASTER_OUTPUT_FILE = "files.txt";
    private static final String TARGET_FILENAME = "files.txt"; // The file to look for on the server

    public static void main(String[] args) {
        try {
            // 1. Setup Directories
            Path mergeDirPath = Paths.get(MERGE_DIR);
            if (!Files.exists(mergeDirPath)) {
                Files.createDirectories(mergeDirPath);
            }

            // 2. Read Servers List
            List<String> servers = readServersList();
            if (servers.isEmpty()) {
                System.out.println("No servers found in " + SERVERS_LIST_FILE);
                return;
            }

            // 3. Process each server (Download & Save)
            System.out.println("--- Starting Downloads ---");
            for (String serverUrl : servers) {
                processServer(serverUrl, mergeDirPath);
            }

            // 4. Merge Files
            System.out.println("\n--- Starting Merge Process ---");
            mergeDownloadedFiles(mergeDirPath);

            System.out.println("\nProcess Completed Successfully.");

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static List<String> readServersList() {
        try {
            Path path = Paths.get(SERVERS_LIST_FILE);
            if (!Files.exists(path)) return Collections.emptyList();
            return Files.readAllLines(path).stream()
                    .map(String::trim)
                    .filter(line -> !line.isEmpty())
                    .collect(Collectors.toList());
        } catch (IOException e) {
            System.err.println("Error reading servers list: " + e.getMessage());
            return Collections.emptyList();
        }
    }

    private static void processServer(String baseUrl, Path outputDir) {
        try {
            // Ensure URL ends with slash before appending filename
            String fullUrlStr = baseUrl.endsWith("/") ? baseUrl + TARGET_FILENAME : baseUrl + "/" + TARGET_FILENAME;
            URL url = new URL(fullUrlStr);

            // Get IP Address
            InetAddress address = InetAddress.getByName(url.getHost());
            String ipAddress = address.getHostAddress();

            // Generate SHA256 of IP
            String ipHash = calculateSha256(ipAddress);
            Path outputPath = outputDir.resolve(ipHash + ".txt");

            System.out.println("Processing: " + baseUrl + " -> IP: " + ipAddress + " -> Hash: " + ipHash);

            // Download and Save
            try (InputStream in = url.openStream()) {
                Files.copy(in, outputPath, StandardCopyOption.REPLACE_EXISTING);
                System.out.println("  [OK] Saved to " + outputPath.getFileName());
            }

        } catch (UnknownHostException e) {
            System.err.println("  [ERR] Could not resolve IP for: " + baseUrl);
        } catch (IOException e) {
            System.err.println("  [ERR] Failed to download from: " + baseUrl + " (" + e.getMessage() + ")");
        }
    }

    private static void mergeDownloadedFiles(Path mergeDirPath) throws IOException {
        Path masterFile = Paths.get(MASTER_OUTPUT_FILE);
        
        // Load existing lines from master file to avoid duplicates
        Set<String> existingLines = new HashSet<>();
        if (Files.exists(masterFile)) {
            existingLines.addAll(Files.readAllLines(masterFile));
        }

        // We use a temporary list to hold new unique lines to append
        List<String> linesToAppend = new ArrayList<>();

        // Iterate over all files in "merge_files" directory
        try (Stream<Path> stream = Files.list(mergeDirPath)) {
            stream.filter(file -> !Files.isDirectory(file))
                  .forEach(path -> {
                      try {
                          List<String> fileLines = Files.readAllLines(path);
                          for (String line : fileLines) {
                              String trimmed = line.trim();
                              // Only add if it's not empty, not already in file, and not already staged for append
                              if (!trimmed.isEmpty() && !existingLines.contains(trimmed)) {
                                  linesToAppend.add(trimmed);
                                  existingLines.add(trimmed); // Add to set so we don't add it twice in this run
                              }
                          }
                      } catch (IOException e) {
                          System.err.println("Error reading downloaded file: " + path);
                      }
                  });
        }

        // Append new unique lines to files.txt
        if (!linesToAppend.isEmpty()) {
            StandardOpenOption option = Files.exists(masterFile) ? StandardOpenOption.APPEND : StandardOpenOption.CREATE;
            Files.write(masterFile, linesToAppend, StandardCharsets.UTF_8, option);
            System.out.println("Added " + linesToAppend.size() + " new unique lines to " + MASTER_OUTPUT_FILE);
        } else {
            System.out.println("No new unique lines found to merge.");
        }
    }

    private static String calculateSha256(String input) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] encodedhash = digest.digest(input.getBytes(StandardCharsets.UTF_8));
            
            // Convert byte array to Hex String
            StringBuilder hexString = new StringBuilder(2 * encodedhash.length);
            for (byte b : encodedhash) {
                String hex = Integer.toHexString(0xff & b);
                if (hex.length() == 1) {
                    hexString.append('0');
                }
                hexString.append(hex);
            }
            return hexString.toString();
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("SHA-256 algorithm not found", e);
        }
    }
}