import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

public class ServerCrawler {

    // Set to keep track of unique HOSTS we have checked
    // This prevents checking "https://app.test.com" twice, but allows "https://test.com" separately
    private static final Set<String> checkedHosts = ConcurrentHashMap.newKeySet();
    
    // Queue for crawling
    private static final Queue<String> urlQueue = new LinkedList<>();

    private static final String INPUT_FILE = "servers.txt";
    private static final String OUTPUT_FILE = "tmp_servers.txt";

    // HttpClient configuration
    private static final HttpClient client = HttpClient.newBuilder()
            .version(HttpClient.Version.HTTP_2)
            .connectTimeout(Duration.ofSeconds(4))
            .followRedirects(HttpClient.Redirect.NORMAL)
            .build();

    public static void main(String[] args) {
        System.out.println("--- Starting Server Crawler (Subdomain Aware) ---");

        // 1. Initialize from file
        try {
            Path inputPath = Paths.get(INPUT_FILE);
            if (Files.exists(inputPath)) {
                List<String> initialUrls = Files.readAllLines(inputPath);
                urlQueue.addAll(initialUrls);
            } else {
                System.out.println("Error: " + INPUT_FILE + " not found.");
                waitForEnter();
                return;
            }
            
            // Clear or create the output file
            Files.writeString(Paths.get(OUTPUT_FILE), "", StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);

        } catch (IOException e) {
            e.printStackTrace();
            waitForEnter();
            return;
        }

        // 2. Process the queue
        while (!urlQueue.isEmpty()) {
            String rawUrl = urlQueue.poll();
            if (rawUrl == null || rawUrl.isBlank()) continue;

            processUrl(rawUrl.trim());
        }

        System.out.println("\n--- Crawling Finished ---");
        System.out.println("Results saved in " + OUTPUT_FILE);
        waitForEnter();
    }

    private static void processUrl(String rawUrl) {
        try {
            // Step 1: Normalize to Base URL (Protocol + Host + Subdomain)
            // Removes /index.php, but KEEPS app.test.com
            String baseUrl = normalizeToBaseUrl(rawUrl);

            // Check if this specific host has already been processed
            if (checkedHosts.contains(baseUrl)) {
                return;
            }
            checkedHosts.add(baseUrl); // Mark as visited

            System.out.print("Scanning: " + baseUrl + " ... ");

            // Step 2: Check if this specific Base URL is Online
            if (isOnline(baseUrl)) {
                System.out.print("[ONLINE]");
                
                // Save exactly as found (but without path)
                appendToFile(baseUrl);

                // Step 3: Check for "servers.txt" on this specific host
                String targetFileUrl = baseUrl + "/servers.txt";
                HttpResponse<String> fileResponse = fetchUrl(targetFileUrl);

                if (fileResponse != null && fileResponse.statusCode() == 200) {
                    System.out.println(" -> Found servers.txt");
                    
                    // Parse the file and add new URLs to queue
                    String content = fileResponse.body();
                    Scanner scanner = new Scanner(content);
                    while (scanner.hasNextLine()) {
                        String line = scanner.nextLine().trim();
                        if (!line.isEmpty() && isValidUrl(line)) {
                            urlQueue.add(line);
                        }
                    }
                    scanner.close();
                } else {
                    System.out.println(" -> No servers.txt");
                }
            } else {
                System.out.println("[OFFLINE]");
            }

        } catch (Exception e) {
            System.out.println("[Error: " + e.getMessage() + "]");
        }
    }

    private static boolean isOnline(String url) {
        HttpResponse<String> response = fetchUrl(url);
        return response != null && (response.statusCode() >= 200 && response.statusCode() < 300);
    }

    private static HttpResponse<String> fetchUrl(String url) {
        try {
            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(url))
                    .timeout(Duration.ofSeconds(4))
                    .GET()
                    .build();
            return client.send(request, HttpResponse.BodyHandlers.ofString());
        } catch (Exception e) {
            return null;
        }
    }

    private static void appendToFile(String line) {
        try {
            Files.writeString(
                    Paths.get(OUTPUT_FILE), 
                    line + System.lineSeparator(), 
                    StandardOpenOption.CREATE, 
                    StandardOpenOption.APPEND
            );
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Normalizes the URL to Protocol + Host (including subdomains).
     * e.g. "https://app.test.com/folder/index.php" -> "https://app.test.com"
     */
    private static String normalizeToBaseUrl(String urlString) throws URISyntaxException {
        if (!urlString.startsWith("http")) {
            urlString = "https://" + urlString;
        }
        
        URI uri = new URI(urlString);
        String protocol = uri.getScheme();
        String host = uri.getHost();

        // If something went wrong parsing, return original string as fallback
        if (host == null) return urlString;

        // Return strictly Protocol + Host (stripping path/query)
        return protocol + "://" + host;
    }

    private static boolean isValidUrl(String url) {
        return url != null && (url.startsWith("http://") || url.startsWith("https://"));
    }

    private static void waitForEnter() {
        System.out.println("\nPress [ENTER] to exit...");
        try {
            System.in.read();
        } catch (IOException e) {
            // Ignored
        }
    }
}