import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.util.*;
import java.util.stream.Collectors;

public class JsonConsensusDownloader {

    private static final long MAX_SIZE_BYTES = 500 * 1024; // 500KB
    private static final String OUTPUT_DIR = "json";
    private static final String INPUT_FILE = "files.txt";

    public static void main(String[] args) {
        try {
            // 1. Setup Output Directory
            Path outputDirPath = Paths.get(OUTPUT_DIR);
            if (!Files.exists(outputDirPath)) {
                Files.createDirectories(outputDirPath);
            }

            // 2. Read and Transform URLs
            List<String> lines = Files.readAllLines(Paths.get(INPUT_FILE));
            
            // Map key: Target Filename (e.g., "1234.json")
            // Map value: List of full URLs that point to this file
            Map<String, List<String>> fileGroups = new HashMap<>();

            for (String line : lines) {
                if (line.trim().isEmpty()) continue;
                
                String transformedUrl = transformUrl(line.trim());
                String fileName = getFileNameFromUrl(transformedUrl);

                fileGroups.computeIfAbsent(fileName, k -> new ArrayList<>()).add(transformedUrl);
            }

            // 3. Process each file group
            HttpClient client = HttpClient.newBuilder()
                    .connectTimeout(Duration.ofSeconds(10))
                    .build();

            for (Map.Entry<String, List<String>> entry : fileGroups.entrySet()) {
                String targetFileName = entry.getKey();
                List<String> urls = entry.getValue();
                
                System.out.println("Processing group: " + targetFileName + " (" + urls.size() + " sources)");
                processFileGroup(client, targetFileName, urls, outputDirPath);
            }

            System.out.println("Processing complete.");

        } catch (IOException e) {
            System.err.println("Error reading files.txt: " + e.getMessage());
        }
    }

    /**
     * Transforms URL: /files/ -> /json/ and extension -> .json
     */
    private static String transformUrl(String originalUrl) {
        // Replace folder
        String newUrl = originalUrl.replace("/files/", "/json/");
        
        // Replace extension
        int lastDotIndex = newUrl.lastIndexOf('.');
        if (lastDotIndex > 0) {
            newUrl = newUrl.substring(0, lastDotIndex) + ".json";
        } else {
            // Edge case: URL has no extension, append .json
            newUrl = newUrl + ".json";
        }
        return newUrl;
    }

    private static String getFileNameFromUrl(String url) {
        return url.substring(url.lastIndexOf('/') + 1);
    }

    /**
     * Downloads content from all URLs, votes on majority, and saves file.
     */
    private static void processFileGroup(HttpClient client, String fileName, List<String> urls, Path outputDir) {
        // Map ContentString -> Count
        Map<String, Integer> contentVotes = new HashMap<>();

        for (String url : urls) {
            String content = downloadContent(client, url);
            if (content != null) {
                contentVotes.put(content, contentVotes.getOrDefault(content, 0) + 1);
            }
        }

        if (contentVotes.isEmpty()) {
            System.out.println("  - No valid content downloaded for " + fileName);
            return;
        }

        // Determine Winner
        String winningContent = null;
        boolean isTie = false;
        int maxVotes = -1;

        // Sort entries by votes (descending)
        List<Map.Entry<String, Integer>> sortedVotes = new ArrayList<>(contentVotes.entrySet());
        sortedVotes.sort(Map.Entry.<String, Integer>comparingByValue().reversed());

        Map.Entry<String, Integer> firstPlace = sortedVotes.get(0);
        maxVotes = firstPlace.getValue();
        winningContent = firstPlace.getKey();

        // Check for tie with the second place
        if (sortedVotes.size() > 1) {
            Map.Entry<String, Integer> secondPlace = sortedVotes.get(1);
            if (secondPlace.getValue() == maxVotes) {
                isTie = true;
                // Randomly choose between the tied options
                if (new Random().nextBoolean()) {
                    winningContent = secondPlace.getKey();
                }
                // (Otherwise keep firstPlace as winner)
            }
        }

        // Determine final filename
        String finalName = fileName;
        if (isTie) {
            // Inject "_temp" before extension
            int dotIndex = fileName.lastIndexOf('.');
            if (dotIndex != -1) {
                finalName = fileName.substring(0, dotIndex) + "_temp" + fileName.substring(dotIndex);
            } else {
                finalName = fileName + "_temp";
            }
            System.out.println("  - Tie detected. Saving as " + finalName);
        } else {
            System.out.println("  - Majority found (" + maxVotes + " votes). Saving as " + finalName);
        }

        // Save to disk
        try {
            Files.writeString(outputDir.resolve(finalName), winningContent, 
                StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        } catch (IOException e) {
            System.err.println("  - Error writing file: " + e.getMessage());
        }
    }

    /**
     * Downloads content strictly adhering to 500KB limit.
     * Returns content String or null if failed/too large.
     */
    private static String downloadContent(HttpClient client, String url) {
        try {
            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(url))
                    .GET()
                    .build();

            HttpResponse<byte[]> response = client.send(request, HttpResponse.BodyHandlers.ofByteArray());

            if (response.statusCode() == 200) {
                byte[] data = response.body();
                
                // Size Check
                if (data.length > MAX_SIZE_BYTES) {
                    System.out.println("    - [Skipped] File too large (" + (data.length/1024) + "kb): " + url);
                    return null;
                }
                
                return new String(data, StandardCharsets.UTF_8);
            }
        } catch (Exception e) {
            System.out.println("    - [Error] Failed to download " + url);
        }
        return null;
    }
}