import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.stream.Collectors;

public class FileRepeatCounterDescending {

    private static final String FILENAME_TO_READ = "files.txt";
    private static final int MAX_RESULTS = 100;

    public static void main(String[] args) {
        // 1. Map to store filenames and their counts
        Map<String, Integer> fileCounts = new HashMap<>();

        // 2. Read the file and populate the map
        System.out.println("Reading file: " + FILENAME_TO_READ + " and counting occurrences...");
        try (BufferedReader br = new BufferedReader(new FileReader(FILENAME_TO_READ))) {
            String line;
            while ((line = br.readLine()) != null) {
                String filename = extractFilename(line);
                if (!filename.isEmpty()) {
                    fileCounts.put(filename, fileCounts.getOrDefault(filename, 0) + 1);
                }
            }
        } catch (IOException e) {
            System.err.println("An error occurred while reading the file: " + e.getMessage());
            // Exit if the file cannot be read
            return;
        }

        // 3. Sort the map by count in descending order
        // This puts the highest counts at the beginning of the stream.
        Map<String, Integer> sortedByCount = fileCounts.entrySet()
                .stream()
                // *** KEY CHANGE HERE ***
                // Use Comparator.reverseOrder() to sort by count in descending order
                .sorted(Map.Entry.comparingByValue(Comparator.reverseOrder())) 
                .limit(MAX_RESULTS) // Limit to the top 100 results (most repeated)
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        Map.Entry::getValue,
                        (oldValue, newValue) -> oldValue, 
                        LinkedHashMap::new // Use LinkedHashMap to maintain the sorting order
                ));

        // 4. Print the results
        System.out.println("\n--- Top " + MAX_RESULTS + " Most Repeated Files (in descending order of count) ---");
        sortedByCount.forEach((filename, count) -> {
            System.out.println(filename + ", " + count);
        });
        System.out.println("---------------------------------------------------------------------");
    }

    /**
     * Extracts the filename from a URL or file path.
     */
    private static String extractFilename(String url) {
        if (url == null || url.isEmpty()) {
            return "";
        }
        int lastSlashIndex = url.lastIndexOf('/');
        
        if (lastSlashIndex != -1 && lastSlashIndex < url.length() - 1) {
            return url.substring(lastSlashIndex + 1).trim();
        }
        return url.trim();
    }
}