import java.io.*;
import java.net.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.*;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;

public class FileHashValidator {

    private static final String FILES_LIST = "files.txt";
    private static final String SERVERS_LIST = "servers.txt";
    private static final String DB_DIR = "database";

    public static void main(String[] args) {
        System.out.println("Starting File Validator...");

        List<String> filenames = readLines(FILES_LIST);
        List<String> servers = readLines(SERVERS_LIST);

        if (filenames.isEmpty() || servers.isEmpty()) {
            System.err.println("Error: Ensure files.txt and servers.txt exist and are not empty.");
            return;
        }

        // Create database directory if it doesn't exist
        File dbDir = new File(DB_DIR);
        if (!dbDir.exists()) {
            dbDir.mkdir();
        }

        for (String server : servers) {
            // Normalize server URL (remove trailing slash if present)
            String baseUrl = server.endsWith("/") ? server.substring(0, server.length() - 1) : server;

            for (String filename : filenames) {
                String fullUrl = baseUrl + "/files/" + filename;
                System.out.println("Checking: " + fullUrl);

                processUrl(fullUrl, filename);
            }
        }
        System.out.println("Process complete.");
    }

    private static void processUrl(String urlString, String filename) {
        HttpURLConnection connection = null;
        try {
            URL url = new URL(urlString);
            connection = (HttpURLConnection) url.openConnection();
            connection.setRequestMethod("GET");
            connection.setConnectTimeout(5000); // 5 seconds timeout
            connection.setReadTimeout(5000);

            // Check if file exists (HTTP 200 OK)
            if (connection.getResponseCode() == HttpURLConnection.HTTP_OK) {
                
                // 1. Calculate Hash of the downloaded file content
                String fileHash;
                try (InputStream is = connection.getInputStream()) {
                    fileHash = calculateStreamHash(is);
                }

                // 2. Remove extension from filename to compare
                String filenameNoExt = getFileNameWithoutExtension(filename);

                // 3. Compare Hashes
                if (fileHash.equalsIgnoreCase(filenameNoExt)) {
                    System.out.println(" >> MATCH FOUND! Processing match...");
                    
                    // 4. Get IP Address of the URL Host
                    InetAddress address = InetAddress.getByName(url.getHost());
                    String ipAddress = address.getHostAddress();
                    
                    // 5. Hash the IP Address
                    String ipHash = calculateStringHash(ipAddress);

                    // 6. Create Directory (Name = File Hash)
                    Path folderPath = Paths.get(DB_DIR, fileHash);
                    if (!Files.exists(folderPath)) {
                        Files.createDirectories(folderPath);
                    }

                    // 7. Create Text File (Name = IP Hash)
                    Path filePath = folderPath.resolve(ipHash + ".txt");
                    
                    // 8. Write URL to file
                    Files.write(filePath, urlString.getBytes(StandardCharsets.UTF_8));
                    
                    System.out.println(" >> Database entry created: " + filePath.toString());
                } else {
                    System.out.println(" >> Hash mismatch. Server file: " + fileHash + " vs Name: " + filenameNoExt);
                }
            } else {
                System.out.println(" >> File not found on server (HTTP " + connection.getResponseCode() + ")");
            }

        } catch (Exception e) {
            System.err.println(" >> Error processing " + urlString + ": " + e.getMessage());
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
    }

    // --- Helper Methods ---

    // Reads lines from a text file
    private static List<String> readLines(String fileName) {
        List<String> lines = new ArrayList<>();
        try (BufferedReader br = new BufferedReader(new FileReader(fileName))) {
            String line;
            while ((line = br.readLine()) != null) {
                if (!line.trim().isEmpty()) {
                    lines.add(line.trim());
                }
            }
        } catch (IOException e) {
            System.err.println("Could not read " + fileName + ": " + e.getMessage());
        }
        return lines;
    }

    // Calculates SHA-256 of an Input Stream
    private static String calculateStreamHash(InputStream is) throws NoSuchAlgorithmException, IOException {
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        byte[] buffer = new byte[8192];
        int bytesRead;
        while ((bytesRead = is.read(buffer)) != -1) {
            digest.update(buffer, 0, bytesRead);
        }
        return bytesToHex(digest.digest());
    }

    // Calculates SHA-256 of a String
    private static String calculateStringHash(String input) throws NoSuchAlgorithmException {
        MessageDigest digest = MessageDigest.getInstance("SHA-256");
        byte[] encodedhash = digest.digest(input.getBytes(StandardCharsets.UTF_8));
        return bytesToHex(encodedhash);
    }

    // Converts byte array to Hex String
    private static String bytesToHex(byte[] hash) {
        StringBuilder hexString = new StringBuilder(2 * hash.length);
        for (byte b : hash) {
            String hex = Integer.toHexString(0xff & b);
            if (hex.length() == 1) {
                hexString.append('0');
            }
            hexString.append(hex);
        }
        return hexString.toString();
    }

    // Removes extension from filename
    private static String getFileNameWithoutExtension(String filename) {
        int dotIndex = filename.lastIndexOf('.');
        if (dotIndex == -1) {
            return filename;
        }
        return filename.substring(0, dotIndex);
    }
}