import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.List;

public class FileDownloader {

    // Nome do arquivo de entrada e diretrio de sada
    private static final String INPUT_FILE = "files.txt";
    private static final String OUTPUT_DIR = "files";

    public static void main(String[] args) {
        // 1. Criar o diretrio de sada se no existir
        Path outputDirPath = Paths.get(OUTPUT_DIR);
        if (!createDirectoryIfNotExists(outputDirPath)) {
            return; // Encerra se no conseguir criar o diretrio
        }

        // 2. Ler o arquivo files.txt e processar cada linha
        Path inputFilePath = Paths.get(INPUT_FILE);
        
        if (!Files.exists(inputFilePath)) {
            System.err.println("Erro: O arquivo '" + INPUT_FILE + "' no foi encontrado.");
            return;
        }

        try {
            List<String> urls = Files.readAllLines(inputFilePath);
            System.out.println("Iniciando download de " + urls.size() + " arquivos...");

            int successCount = 0;
            for (String url : urls) {
                // Ignora linhas vazias
                if (url.trim().isEmpty()) continue;
                
                if (downloadFile(url.trim(), outputDirPath)) {
                    successCount++;
                }
            }

            System.out.println("------------------------------------------------");
            System.out.println("Processo finalizado. " + successCount + "/" + urls.size() + " arquivos baixados com sucesso.");

        } catch (IOException e) {
            System.err.println("Erro ao ler o arquivo files.txt: " + e.getMessage());
        }
    }

    /**
     * Tenta baixar um nico arquivo dado uma URL string.
     */
    private static boolean downloadFile(String urlString, Path outputDir) {
        try {
            URL url = new URL(urlString);
            
            // Extrai o nome do arquivo da URL (ex: "imagem.jpg" de "http://site.com/imagem.jpg")
            String fileName = extractFileName(urlString);
            Path targetPath = outputDir.resolve(fileName);

            System.out.print("Baixando: " + fileName + " ... ");

            // Abre o stream da URL e copia para o arquivo local
            try (InputStream in = url.openStream()) {
                Files.copy(in, targetPath, StandardCopyOption.REPLACE_EXISTING);
                System.out.println("[OK]");
                return true;
            }

        } catch (Exception e) {
            System.out.println("[FALHA]");
            System.err.println(" -> Erro ao baixar " + urlString + ": " + e.getMessage());
            return false;
        }
    }

    /**
     * Cria o diretrio de destino.
     */
    private static boolean createDirectoryIfNotExists(Path path) {
        if (!Files.exists(path)) {
            try {
                Files.createDirectories(path);
                System.out.println("Diretrio '" + OUTPUT_DIR + "' criado.");
            } catch (IOException e) {
                System.err.println("Erro crtico: No foi possvel criar o diretrio 'files'.");
                return false;
            }
        }
        return true;
    }

    /**
     * Extrai o nome do arquivo da URL de forma segura.
     * Remove query params (?id=1) se existirem.
     */
    private static String extractFileName(String urlString) {
        try {
            // Usa URI para fazer o parse correto do caminho
            String path = new URI(urlString).getPath();
            Path p = Paths.get(path);
            return p.getFileName().toString();
        } catch (Exception e) {
            // Fallback simples caso o URI parse falhe
            return urlString.substring(urlString.lastIndexOf('/') + 1);
        }
    }
}