﻿<?php
/**
 * Rhyzom Project - Landing Page & Data Processor
 * All-in-one file: Logic, Landing Page, and Application.
 */

$page = isset($_GET['page']) ? $_GET['page'] : 'landing';
$message = "";
$messageType = "";

// --- LOGIC SECTION ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && $page == 'app') {
    try {
        // 1. Process Categories
        $rawCategories = isset($_POST['categories']) ? $_POST['categories'] : '';
        $categoriesArray = array_filter(array_map('trim', explode(',', $rawCategories)));
        
        if (count($categoriesArray) > 5) {
            throw new Exception("Error: Maximum of 5 categories allowed.");
        }

        foreach ($categoriesArray as $cat) {
            if (strlen($cat) > 12) {
                throw new Exception("Error: Category '$cat' exceeds 12 characters.");
            }
            if (!preg_match('/^[a-zA-Z0-9_]+$/', $cat)) {
                throw new Exception("Error: Category '$cat' contains invalid characters. Use letters, numbers, and '_' only.");
            }
        }

        // 2. Process Signature and Hash
        $signature = isset($_POST['signature']) ? trim($_POST['signature']) : '';
        if (strlen($signature) > 50) {
            throw new Exception("Error: Signature must be maximum 50 characters.");
        }
        if (empty($signature)) {
            throw new Exception("Error: Signature is required.");
        }
        
        $signatureHash = hash('sha256', $signature);

        // 3. Process Textarea (Metadata and URLs)
        $contentLines = isset($_POST['content']) ? explode("\n", str_replace("\r", "", $_POST['content'])) : [];
        $finalContent = "";
        
        if (!empty($contentLines)) {
            // First line: Metadata (BTC or other)
            $finalContent .= $contentLines[0] . "\n";
            
            // Subsequent lines: URL Validation
            for ($i = 1; $i < count($contentLines); $i++) {
                $line = trim($contentLines[$i]);
                if (!empty($line)) {
                    if (filter_var($line, FILTER_VALIDATE_URL)) {
                        $finalContent .= $line . "\n";
                    } else {
                        throw new Exception("Error: Line " . ($i + 1) . " is not a valid URL.");
                    }
                }
            }
        }

        // 4. Save Files
        $baseDir = "rhyzom";
        if (!is_dir($baseDir)) {
            mkdir($baseDir, 0777, true);
        }

        foreach ($categoriesArray as $category) {
            $catDir = $baseDir . DIRECTORY_SEPARATOR . $category;
            if (!is_dir($catDir)) {
                mkdir($catDir, 0777, true);
            }

            $filePath = $catDir . DIRECTORY_SEPARATOR . $signatureHash . ".txt";
            
            $handle = fopen($filePath, "w");
            if ($handle) {
                fwrite($handle, $finalContent);
                fclose($handle);
            } else {
                throw new Exception("Error opening file for writing at: $filePath");
            }
        }

        $message = "Success! Files processed and saved inside 'rhyzom' folder.";
        $messageType = "success";

    } catch (Exception $e) {
        $message = $e->getMessage();
        $messageType = "error";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rhyzom - Secure Data Categorization</title>
    <style>
        :root {
            --primary: #2563eb;
            --primary-hover: #1d4ed8;
            --success: #059669;
            --error: #dc2626;
            --bg: #f8fafc;
            --text: #1e293b;
            --card: #ffffff;
        }
        body {
            font-family: 'Inter', -apple-system, sans-serif;
            background-color: var(--bg);
            color: var(--text);
            margin: 0;
            line-height: 1.6;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            padding: 2rem;
        }
        nav {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1rem 0;
            border-bottom: 1px solid #e2e8f0;
            margin-bottom: 2rem;
        }
        .logo {
            font-size: 1.5rem;
            font-weight: 800;
            color: var(--primary);
            text-decoration: none;
        }
        .nav-links a {
            text-decoration: none;
            color: var(--text);
            font-weight: 500;
            margin-left: 1.5rem;
            transition: color 0.2s;
        }
        .nav-links a:hover {
            color: var(--primary);
        }
        
        /* Landing Page Styles */
        .hero {
            text-align: center;
            padding: 4rem 0;
        }
        .hero h1 {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: #0f172a;
        }
        .hero p {
            font-size: 1.25rem;
            color: #64748b;
            max-width: 600px;
            margin: 0 auto 2rem;
        }
        .cta-button {
            display: inline-block;
            background: var(--primary);
            color: white;
            padding: 1rem 2rem;
            border-radius: 0.5rem;
            text-decoration: none;
            font-weight: 600;
            transition: background 0.2s;
        }
        .cta-button:hover {
            background: var(--primary-hover);
        }
        .features {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 2rem;
            margin-top: 4rem;
        }
        .feature-card {
            background: var(--card);
            padding: 2rem;
            border-radius: 1rem;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }
        .feature-card h3 {
            margin-top: 0;
            color: var(--primary);
        }

        /* App Styles */
        .app-card {
            background: var(--card);
            padding: 2.5rem;
            border-radius: 1rem;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
        }
        .form-group {
            margin-bottom: 1.5rem;
        }
        label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
        }
        input[type="text"], textarea {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #cbd5e1;
            border-radius: 0.375rem;
            box-sizing: border-box;
            font-size: 1rem;
            font-family: inherit;
        }
        textarea {
            height: 180px;
        }
        small {
            color: #64748b;
            display: block;
            margin-top: 0.4rem;
        }
        .submit-btn {
            width: 100%;
            padding: 1rem;
            background: var(--success);
            color: white;
            border: none;
            border-radius: 0.375rem;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: opacity 0.2s;
        }
        .submit-btn:hover {
            opacity: 0.9;
        }
        .alert {
            padding: 1rem;
            border-radius: 0.375rem;
            margin-bottom: 1.5rem;
            font-weight: 500;
        }
        .alert-success { background: #dcfce7; color: #166534; border: 1px solid #bbf7d0; }
        .alert-error { background: #fee2e2; color: #991b1b; border: 1px solid #fecaca; }

        footer {
            text-align: center;
            padding: 4rem 0 2rem;
            color: #94a3b8;
            font-size: 0.875rem;
        }
    </style>
</head>
<body>

<div class="container">
    <nav>
        <a href="?page=landing" class="logo">RHYZOM</a>
        <div class="nav-links">
            <a href="?page=landing">Home</a>
            <a href="index_simple.php">More</a>
        </div>
    </nav>

    <?php if ($page == 'landing'): ?>
        <!-- LANDING PAGE -->
        <section class="hero">
            <h1>Organize with Privacy</h1>
            <p>A lightweight, secure tool to categorize your metadata and URLs using cryptographic signatures.</p>
            <a href="?page=app" class="cta-button">Get Started Now</a>
        </section>

        <section class="features">
            <div class="feature-card">
                <h3>Smart Categorization</h3>
                <p>Organize your data into up to 5 custom categories. Each category creates a dedicated workspace in the file system.</p>
            </div>
            <div class="feature-card">
                <h3>Crypto-Naming</h3>
                <p>Your files are named using SHA-256 hashes of your signature, ensuring privacy and preventing unauthorized file discovery.</p>
            </div>
            <div class="feature-card">
                <h3>URL Validation</h3>
                <p>Built-in verification ensures that every link you save is a valid URL, while keeping your metadata safe on the first line.</p>
            </div>
        </section>

        <section style="margin-top: 4rem; background: #fff; padding: 2rem; border-radius: 1rem;">
            <h2>How it works?</h2>
            <ol>
                <li><strong>Define Categories:</strong> Enter up to 5 keywords (e.g., <code>work, crypto, personal</code>).</li>
                <li><strong>Sign your Data:</strong> Provide a signature (up to 50 chars). This becomes your unique file identifier via SHA-256.</li>
                <li><strong>Input Content:</strong> Put your BTC address or metadata on the first line, followed by your URLs.</li>
                <li><strong>Save:</strong> The system creates a folder structure <code>/rhyzom/[category]/[hash].txt</code> automatically.</li>
            </ol>
        </section>

    <?php else: ?>
        <!-- APP PAGE -->
        <div class="app-card">
            <h2 style="margin-top: 0;">Data Processor</h2>
            
            <?php if ($message): ?>
                <div class="alert alert-<?php echo $messageType; ?>">
                    <?php echo $message; ?>
                </div>
            <?php endif; ?>

            <form method="POST" action="?page=app">
                <div class="form-group">
                    <label for="categories">Categories</label>
                    <input type="text" id="categories" name="categories" placeholder="e.g. tech, news, btc" required>
                    <small>Max 5 categories, comma separated. Max 12 chars each (A-Z, 0-9, _).</small>
                </div>

                <div class="form-group">
                    <label for="signature">Signature</label>
                    <input type="text" id="signature" name="signature" maxlength="50" placeholder="Your secret signature" required>
                    <small>This will be hashed (SHA-256) to generate the filename.</small>
                </div>

                <div class="form-group">
                    <label for="content">Content</label>
                    <textarea id="content" name="content" placeholder="Line 1: BTC Address or Metadata&#10;Line 2+: https://example.com&#10;Line 3+: https://another.com" required></textarea>
                    <small>First line is metadata. All following lines must be valid URLs.</small>
                </div>

                <button type="submit" class="submit-btn">Process & Save Files</button>
            </form>
            
            <p style="text-align: center; margin-top: 1.5rem;">
                <a href="?page=landing" style="color: var(--primary); text-decoration: none;">&larr; Back to Home</a>
            </p>
        </div>
    <?php endif; ?>

    <footer>
        &copy; <?php echo date('Y'); ?> Rhyzom Project. Secure & Minimalist.
    </footer>
</div>

</body>
</html>
