<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>File Hash & Description Saver</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        body {
            background-color: #f5f7fa;
            color: #333;
            line-height: 1.6;
            padding: 20px;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 30px;
        }

        @media (max-width: 768px) {
            .container {
                grid-template-columns: 1fr;
            }
        }

        .main-section {
            background-color: white;
            border-radius: 10px;
            padding: 30px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }

        .sidebar {
            background-color: white;
            border-radius: 10px;
            padding: 30px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }

        h1 {
            color: #2c3e50;
            margin-bottom: 10px;
            border-bottom: 3px solid #3498db;
            padding-bottom: 10px;
        }

        h2 {
            color: #2c3e50;
            margin: 20px 0 15px;
            font-size: 1.4rem;
        }

        .file-upload-area {
            border: 2px dashed #3498db;
            border-radius: 8px;
            padding: 40px 20px;
            text-align: center;
            margin: 20px 0;
            background-color: #f8fafd;
            transition: all 0.3s;
            cursor: pointer;
        }

        .file-upload-area:hover {
            background-color: #edf7ff;
            border-color: #2980b9;
        }

        .file-upload-area.dragover {
            background-color: #e1f0ff;
            border-color: #1abc9c;
            border-style: solid;
        }

        .file-info {
            margin: 15px 0;
            padding: 15px;
            background-color: #f8f9fa;
            border-radius: 6px;
            display: none;
        }

        .file-info.active {
            display: block;
        }

        textarea {
            width: 100%;
            padding: 15px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 1rem;
            resize: vertical;
            min-height: 150px;
            margin: 10px 0 20px;
        }

        .char-count {
            text-align: right;
            font-size: 0.9rem;
            color: #7f8c8d;
            margin-top: -15px;
            margin-bottom: 15px;
        }

        .char-count.near-limit {
            color: #e67e22;
        }

        .char-count.over-limit {
            color: #e74c3c;
        }

        .btn {
            background-color: #3498db;
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 6px;
            font-size: 1rem;
            cursor: pointer;
            transition: background-color 0.3s;
            margin-top: 10px;
        }

        .btn:hover {
            background-color: #2980b9;
        }

        .btn:disabled {
            background-color: #bdc3c7;
            cursor: not-allowed;
        }

        .result {
            margin-top: 25px;
            padding: 20px;
            border-radius: 6px;
            display: none;
        }

        .result.success {
            background-color: #d5edda;
            color: #155724;
            border: 1px solid #c3e6cb;
            display: block;
        }

        .result.error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
            display: block;
        }

        .hash-display {
            word-break: break-all;
            font-family: monospace;
            background-color: #f1f1f1;
            padding: 10px;
            border-radius: 4px;
            margin-top: 10px;
        }

        .social-section {
            margin-top: 30px;
        }

        .social-links {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 15px;
        }

        .social-link {
            display: flex;
            align-items: center;
            padding: 10px 15px;
            background-color: #f8f9fa;
            border-radius: 6px;
            text-decoration: none;
            color: #333;
            transition: all 0.3s;
            border: 1px solid #e9ecef;
        }

        .social-link:hover {
            background-color: #e9ecef;
            transform: translateY(-2px);
        }

        .social-icon {
            margin-right: 8px;
            font-size: 1.2rem;
        }

        .faq {
            margin-top: 30px;
            background-color: #f8fafd;
            padding: 20px;
            border-radius: 8px;
        }

        .faq h3 {
            margin-bottom: 15px;
            color: #2c3e50;
        }

        .faq-item {
            margin-bottom: 15px;
        }

        .faq-question {
            font-weight: bold;
            color: #3498db;
        }

        .status-indicator {
            display: inline-block;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
        }

        .status-online {
            background-color: #2ecc71;
        }

        .status-offline {
            background-color: #e74c3c;
        }
    </style>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <div class="container">
        <div class="main-section">
            <h1><i class="fas fa-fingerprint"></i> File Hash & Description Saver</h1>
            <p>Upload a file to generate its SHA256 hash and save a description. Each file's hash becomes a unique identifier stored in the "blocks" folder.</p>
            
            <div class="file-upload-area" id="dropArea">
                <i class="fas fa-cloud-upload-alt" style="font-size: 48px; color: #3498db; margin-bottom: 15px;"></i>
                <h2>Drag & Drop Your File Here</h2>
                <p>Or click to browse</p>
                <input type="file" id="fileInput" style="display: none;">
            </div>
            
            <div class="file-info" id="fileInfo">
                <p><strong>Selected File:</strong> <span id="fileName"></span></p>
                <p><strong>File Size:</strong> <span id="fileSize"></span></p>
                <p><strong>File Type:</strong> <span id="fileType"></span></p>
            </div>
            
            <h2>File Description (Max 500 characters)</h2>
            <textarea id="fileDescription" placeholder="Enter a description for your file..."></textarea>
            <div class="char-count" id="charCount">0/500</div>
            
            <button class="btn" id="submitBtn" disabled>Generate Hash & Save Description</button>
            
            <div class="result" id="result"></div>
            
            <div class="hash-display" id="hashDisplay" style="display:none;">
                <strong>File Hash (SHA256):</strong> <span id="hashValue"></span>
            </div>
        </div>
        
        <div class="sidebar">
            <h2><i class="fas fa-share-alt"></i> Social Media</h2>
            <p>Share this tool or learn more about file verification and auctions:</p>
            
            <div class="social-links">
                <a href="https://twitter.com/intent/tweet?text=Check%20out%20this%20File%20Hash%20Generator%20tool" target="_blank" class="social-link">
                    <i class="fab fa-twitter social-icon" style="color: #1DA1F2;"></i> Twitter
                </a>
                <a href="https://www.facebook.com/sharer/sharer.php?u=example.com" target="_blank" class="social-link">
                    <i class="fab fa-facebook-f social-icon" style="color: #1877F2;"></i> Facebook
                </a>
                <a href="https://www.linkedin.com/shareArticle?mini=true&url=example.com&title=File%20Hash%20Generator" target="_blank" class="social-link">
                    <i class="fab fa-linkedin-in social-icon" style="color: #0A66C2;"></i> LinkedIn
                </a>
                <a href="https://github.com" target="_blank" class="social-link">
                    <i class="fab fa-github social-icon" style="color: #333;"></i> GitHub
                </a>
            </div>
            
            <div class="social-section">
                <h2><i class="fas fa-gavel"></i> Learn About File Auctions</h2>
                <p>File auctions are an emerging concept where digital files with unique hashes can be bought, sold, or traded as verified digital assets.</p>
                
                <div class="faq">
                    <h3>Frequently Asked Questions</h3>
                    <div class="faq-item">
                        <div class="faq-question">What are file auctions?</div>
                        <p>File auctions allow users to trade verified digital files using blockchain-like verification. Each file's unique hash serves as its digital fingerprint.</p>
                    </div>
                    <div class="faq-item">
                        <div class="faq-question">Why use SHA256 hashes?</div>
                        <p>SHA256 generates a unique 64-character hash for any file. Even a tiny change creates a completely different hash, ensuring file integrity.</p>
                    </div>
                    <div class="faq-item">
                        <div class="faq-question">How are files verified?</div>
                        <p>By comparing SHA256 hashes. If two files have the same hash, they are identical at the binary level.</p>
                    </div>
                </div>
                
                <div style="margin-top: 20px; padding: 15px; background-color: #e8f4fc; border-radius: 6px;">
                    <h3>Explore Further</h3>
                    <p>To learn more about file auctions and digital asset trading:</p>
                    <ul style="margin-left: 20px; margin-top: 10px;">
                        <li>Check out blockchain-based file marketplaces</li>
                        <li>Research cryptographic file verification</li>
                        <li>Explore NFT (Non-Fungible Token) technologies</li>
                    </ul>
                </div>
            </div>
            
            <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee;">
                <p><i class="fas fa-info-circle"></i> <strong>Note:</strong> Files are not stored on the server, only their hashes and descriptions in the "blocks" folder.</p>
                <p style="margin-top: 10px; font-size: 0.9rem; color: #7f8c8d;">
                    <span class="status-indicator status-online"></span> System Status: Operational
                </p>
            </div>
        </div>
    </div>

    <script>
        // DOM elements
        const fileInput = document.getElementById('fileInput');
        const dropArea = document.getElementById('dropArea');
        const fileInfo = document.getElementById('fileInfo');
        const fileName = document.getElementById('fileName');
        const fileSize = document.getElementById('fileSize');
        const fileType = document.getElementById('fileType');
        const fileDescription = document.getElementById('fileDescription');
        const charCount = document.getElementById('charCount');
        const submitBtn = document.getElementById('submitBtn');
        const result = document.getElementById('result');
        const hashDisplay = document.getElementById('hashDisplay');
        const hashValue = document.getElementById('hashValue');
        
        let selectedFile = null;
        
        // Event listeners for file selection
        dropArea.addEventListener('click', () => fileInput.click());
        
        fileInput.addEventListener('change', (e) => {
            if (e.target.files.length) {
                handleFileSelection(e.target.files[0]);
            }
        });
        
        // Drag and drop events
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            dropArea.addEventListener(eventName, preventDefaults, false);
        });
        
        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }
        
        ['dragenter', 'dragover'].forEach(eventName => {
            dropArea.addEventListener(eventName, highlight, false);
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            dropArea.addEventListener(eventName, unhighlight, false);
        });
        
        function highlight() {
            dropArea.classList.add('dragover');
        }
        
        function unhighlight() {
            dropArea.classList.remove('dragover');
        }
        
        dropArea.addEventListener('drop', (e) => {
            const dt = e.dataTransfer;
            const file = dt.files[0];
            if (file) {
                handleFileSelection(file);
            }
        });
        
        // Handle file selection
        function handleFileSelection(file) {
            selectedFile = file;
            
            // Update file info display
            fileName.textContent = file.name;
            fileSize.textContent = formatFileSize(file.size);
            fileType.textContent = file.type || 'Unknown';
            fileInfo.classList.add('active');
            
            // Enable submit button
            submitBtn.disabled = false;
            
            // Clear any previous results
            result.className = 'result';
            result.style.display = 'none';
            hashDisplay.style.display = 'none';
        }
        
        // Format file size
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
        
        // Character count for description
        fileDescription.addEventListener('input', () => {
            const length = fileDescription.value.length;
            charCount.textContent = `${length}/500`;
            
            // Update character count styling
            charCount.classList.remove('near-limit', 'over-limit');
            if (length > 450 && length <= 500) {
                charCount.classList.add('near-limit');
            } else if (length > 500) {
                charCount.classList.add('over-limit');
            }
            
            // Truncate if over limit
            if (length > 500) {
                fileDescription.value = fileDescription.value.substring(0, 500);
                charCount.textContent = '500/500';
                charCount.classList.add('over-limit');
            }
        });
        
        // Handle form submission
        submitBtn.addEventListener('click', async () => {
            if (!selectedFile) {
                showResult('Please select a file first.', 'error');
                return;
            }
            
            // Check description length
            if (fileDescription.value.length === 0) {
                showResult('Please enter a description for the file.', 'error');
                return;
            }
            
            if (fileDescription.value.length > 500) {
                showResult('Description must be 500 characters or less.', 'error');
                return;
            }
            
            // Disable button and show processing
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
            
            // Create FormData to send to PHP
            const formData = new FormData();
            formData.append('file', selectedFile);
            formData.append('description', fileDescription.value);
            
            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showResult(data.message, 'success');
                    hashValue.textContent = data.hash;
                    hashDisplay.style.display = 'block';
                } else {
                    showResult(data.message, 'error');
                }
            } catch (error) {
                showResult('An error occurred while processing the file: ' + error.message, 'error');
            } finally {
                submitBtn.disabled = false;
                submitBtn.innerHTML = 'Generate Hash & Save Description';
            }
        });
        
        // Show result message
        function showResult(message, type) {
            result.textContent = message;
            result.className = `result ${type}`;
            result.style.display = 'block';
            
            // Scroll to result
            result.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }
    </script>
    
    <?php
    // PHP code for handling file upload and hash generation
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Set content type for JSON response
        header('Content-Type: application/json');
        
        // Check if file was uploaded
        if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_OK) {
            echo json_encode(['success' => false, 'message' => 'No file uploaded or upload error.']);
            exit;
        }
        
        // Check if description is provided
        if (!isset($_POST['description']) || empty(trim($_POST['description']))) {
            echo json_encode(['success' => false, 'message' => 'Description is required.']);
            exit;
        }
        
        // Validate description length
        $description = trim($_POST['description']);
        if (strlen($description) > 500) {
            echo json_encode(['success' => false, 'message' => 'Description must be 500 characters or less.']);
            exit;
        }
        
        // Get file details
        $fileTmpPath = $_FILES['file']['tmp_name'];
        $fileName = $_FILES['file']['name'];
        $fileSize = $_FILES['file']['size'];
        
        // Generate SHA256 hash of the file
        $fileHash = hash_file('sha256', $fileTmpPath);
        
        if (!$fileHash) {
            echo json_encode(['success' => false, 'message' => 'Error generating file hash.']);
            exit;
        }
        
        // Create blocks directory if it doesn't exist
        $blocksDir = 'blocks';
        if (!is_dir($blocksDir)) {
            mkdir($blocksDir, 0755, true);
        }
        
        // Check if file already exists
        $filePath = $blocksDir . '/' . $fileHash;
        
        if (file_exists($filePath)) {
            echo json_encode([
                'success' => false, 
                'message' => 'A file with this hash already exists in the blocks folder.',
                'hash' => $fileHash
            ]);
            exit;
        }
        
        // Prepare data to save
        $data = [
            'hash' => $fileHash,
            'original_filename' => $fileName,
            'file_size' => $fileSize,
            'upload_date' => date('Y-m-d H:i:s'),
            'description' => $description
        ];
        
        // Save to file
        $result = file_put_contents($filePath, json_encode($data, JSON_PRETTY_PRINT));
        
        if ($result === false) {
            echo json_encode(['success' => false, 'message' => 'Error saving file information.']);
            exit;
        }
        
        // Return success response
        echo json_encode([
            'success' => true, 
            'message' => 'File hash generated and description saved successfully!',
            'hash' => $fileHash
        ]);
        exit;
    }
    ?>
</body>
</html>