<?php
// --- PHP BACKEND SECTION ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');

    // 1. Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    $hash = $input['hash'] ?? '';
    $description = $input['description'] ?? '';

    // 2. Validate Hash (Must be valid SHA256 hex string)
    if (!preg_match('/^[a-f0-9]{64}$/i', $hash)) {
        echo json_encode(['success' => false, 'message' => 'Invalid SHA256 hash.']);
        exit;
    }

    // 3. Validate Description (Max 500 chars)
    if (strlen($description) > 500) {
        echo json_encode(['success' => false, 'message' => 'Description exceeds 500 characters.']);
        exit;
    }

    // 4. Create "blocks" directory if it doesn't exist
    $dir = 'blocks';
    if (!is_dir($dir)) {
        if (!mkdir($dir, 0755, true)) {
            echo json_encode(['success' => false, 'message' => 'Failed to create directory.']);
            exit;
        }
    }

    // 5. Check if file exists
    $filepath = $dir . '/' . $hash . '.txt';
    if (file_exists($filepath)) {
        echo json_encode(['success' => false, 'message' => 'File already exists! No changes made.']);
        exit;
    }

    // 6. Save the file
    if (file_put_contents($filepath, $description)) {
        echo json_encode(['success' => true, 'message' => 'Block saved successfully! Filename: ' . $hash]);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to write file.']);
    }
    exit; // Stop script execution so HTML isn't returned with JSON
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>File Hash Locker</title>
    <style>
        /* --- CSS STYLING --- */
        :root {
            --primary: #4a90e2;
            --bg: #f4f4f9;
            --text: #333;
            --card-bg: #fff;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: var(--bg);
            color: var(--text);
            display: flex;
            flex-direction: column;
            align-items: center;
            min-height: 100vh;
            margin: 0;
            padding: 20px;
        }

        .container {
            background: var(--card-bg);
            padding: 2rem;
            border-radius: 12px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            max-width: 500px;
            width: 100%;
            margin-bottom: 2rem;
        }

        h1 { margin-top: 0; color: var(--primary); }
        h3 { margin-bottom: 0.5rem; }

        label { display: block; margin-top: 1rem; font-weight: bold; }

        input[type="file"] {
            margin-top: 0.5rem;
            width: 100%;
        }

        .hash-display {
            background: #eee;
            padding: 8px;
            border-radius: 4px;
            font-family: monospace;
            word-break: break-all;
            font-size: 0.85rem;
            margin-top: 0.5rem;
            min-height: 20px;
            color: #555;
        }

        textarea {
            width: 100%;
            height: 100px;
            margin-top: 0.5rem;
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 4px;
            resize: vertical;
        }

        button {
            background-color: var(--primary);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 6px;
            cursor: pointer;
            width: 100%;
            margin-top: 1.5rem;
            font-size: 1rem;
            transition: background 0.3s;
        }

        button:hover { background-color: #357abd; }
        button:disabled { background-color: #ccc; cursor: not-allowed; }

        .status {
            margin-top: 1rem;
            padding: 10px;
            border-radius: 4px;
            display: none;
            text-align: center;
        }
        .status.success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .status.error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }

        .info-section {
            text-align: center;
            max-width: 500px;
            font-size: 0.9rem;
            color: #666;
        }

        .social-links {
            display: flex;
            justify-content: center;
            gap: 15px;
            margin-top: 10px;
        }

        .social-links a {
            text-decoration: none;
            color: var(--primary);
            font-weight: bold;
        }

        .char-count { font-size: 0.8rem; color: #888; text-align: right; margin-top: 2px; }
    </style>
</head>
<body>

    <div class="container">
        <h1>Block Registry</h1>
        <p>Select a file to calculate its SHA256 hash and save a description to the blockchain-style folder.</p>

        <label for="fileInput">1. Select File</label>
        <input type="file" id="fileInput">

        <label>SHA256 Hash:</label>
        <div id="hashDisplay" class="hash-display">Waiting for file...</div>

        <label for="descInput">2. Enter Description</label>
        <textarea id="descInput" maxlength="500" placeholder="Enter a description (max 500 letters)..."></textarea>
        <div class="char-count"><span id="charCount">0</span>/500</div>

        <button id="submitBtn" disabled>Save Block</button>

        <div id="statusMsg" class="status"></div>
    </div>

    <div class="info-section">
        <h3>Did you know?</h3>
        <p>
            You can learn more about <strong>File Auctions</strong> and how digital assets are traded securely. 
            Auctions allow the highest bidder to own the rights to unique file hashes.
        </p>

        <hr>

        <h4>Follow us on Social Media</h4>
        <div class="social-links">
            <a href="#">Twitter</a>
            <a href="#">Facebook</a>
            <a href="#">LinkedIn</a>
            <a href="#">Instagram</a>
        </div>
    </div>

    <script>
        const fileInput = document.getElementById('fileInput');
        const hashDisplay = document.getElementById('hashDisplay');
        const descInput = document.getElementById('descInput');
        const charCount = document.getElementById('charCount');
        const submitBtn = document.getElementById('submitBtn');
        const statusMsg = document.getElementById('statusMsg');

        let currentHash = null;

        // 1. Monitor File Input & Calculate Hash
        fileInput.addEventListener('change', async function() {
            if (this.files.length === 0) {
                resetUI();
                return;
            }

            const file = this.files[0];
            try {
                hashDisplay.innerText = "Calculating hash...";
                currentHash = await calculateSHA256(file);
                hashDisplay.innerText = currentHash;
                submitBtn.disabled = false;
            } catch (error) {
                console.error(error);
                hashDisplay.innerText = "Error calculating hash.";
            }
        });

        // 2. Monitor Character Count
        descInput.addEventListener('input', function() {
            const len = this.value.length;
            charCount.innerText = len;
        });

        // 3. Handle Submission
        submitBtn.addEventListener('click', async () => {
            const description = descInput.value;

            if (!currentHash) {
                alert("Please select a file first.");
                return;
            }

            submitBtn.disabled = true;
            submitBtn.innerText = "Processing...";

            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        hash: currentHash,
                        description: description
                    })
                });

                const result = await response.json();

                statusMsg.style.display = 'block';
                statusMsg.className = 'status ' + (result.success ? 'success' : 'error');
                statusMsg.innerText = result.message;

            } catch (err) {
                statusMsg.style.display = 'block';
                statusMsg.className = 'status error';
                statusMsg.innerText = "Network error occurred.";
            } finally {
                submitBtn.disabled = false;
                submitBtn.innerText = "Save Block";
            }
        });

        // Helper: Calculate SHA256
        async function calculateSHA256(file) {
            const buffer = await file.arrayBuffer();
            const hashBuffer = await crypto.subtle.digest('SHA-256', buffer);
            const hashArray = Array.from(new Uint8Array(hashBuffer));
            const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
            return hashHex;
        }

        function resetUI() {
            currentHash = null;
            hashDisplay.innerText = "Waiting for file...";
            submitBtn.disabled = true;
            statusMsg.style.display = 'none';
        }
    </script>
</body>
</html>